package ru.yandex.solomon.gateway.backend.storage;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
// TODO: class should not be spring bean
@Component
@ParametersAreNonnullByDefault
public class ReadMetrics implements MetricSupplier {
    private final ConcurrentMap<String, ClientMetrics> metrics = new ConcurrentHashMap<>();

    public ClientMetrics getClientMetrics(String clientId) {
        return metrics.computeIfAbsent(clientId, ClientMetrics::new);
    }

    @Override
    public int estimateCount() {
        return metrics.values()
                .stream()
                .mapToInt(ClientMetrics::estimateCount)
                .sum();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        ClientMetrics total = new ClientMetrics("total");
        for (ClientMetrics shard : metrics.values()) {
            total.combine(shard);
            shard.append(tsMillis, commonLabels, consumer);
        }
        total.append(tsMillis, commonLabels, consumer);
    }

    public static class ClientMetrics implements MetricSupplier {
        private final String client;
        private final ConcurrentMap<String, ShardMetrics> metrics = new ConcurrentHashMap<>();

        public ClientMetrics(String client) {
            this.client = client;
        }

        public ShardMetrics getShardMetrics(String shardId) {
            return metrics.computeIfAbsent(shardId, ShardMetrics::new);
        }

        public ClientMetrics combine(ClientMetrics right) {
            for (Map.Entry<String, ShardMetrics> entry : right.metrics.entrySet()) {
                metrics.computeIfAbsent(entry.getKey(), ShardMetrics::new)
                        .combine(entry.getValue());
            }
            return this;
        }

        @Override
        public int estimateCount() {
            return metrics.values()
                    .stream()
                    .mapToInt(ShardMetrics::estimateCount)
                    .sum();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            commonLabels = commonLabels.add("client", client);
            ShardMetrics total = new ShardMetrics("total");
            for (ShardMetrics shard : metrics.values()) {
                total.combine(shard);
                shard.append(tsMillis, commonLabels, consumer);
            }
            total.append(tsMillis, commonLabels, consumer);
        }
    }

    public static class ShardMetrics implements MetricSupplier {
        private static double[] READ_OFFSET_TIME_MINUTES_BOUNDS = {
                TimeUnit.MINUTES.toMinutes(15),
                TimeUnit.HOURS.toMinutes(1),
                TimeUnit.HOURS.toMinutes(3),
                TimeUnit.HOURS.toMinutes(12),
                TimeUnit.DAYS.toMinutes(1),
                TimeUnit.DAYS.toMinutes(3),
                TimeUnit.DAYS.toMinutes(7),
                TimeUnit.DAYS.toMinutes(31),
                TimeUnit.DAYS.toMinutes(90),
                TimeUnit.DAYS.toMinutes(365)
        };

        private final String shardId;
        private final MetricRegistry registry;

        private final Rate metricsReadStarted;
        private final Rate metricsReadCompleted;
        private final LazyGaugeInt64 metricsReadInflight;
        private final Histogram metricsReadElapsedTime;
        private final Histogram metricsReadOffsetMinutes;

        public ShardMetrics(String shardId) {
            this.shardId = shardId;
            this.registry = new MetricRegistry(Labels.of("shardId", shardId));

            this.metricsReadStarted = registry.rate("sensors.read.started");
            this.metricsReadCompleted = registry.rate("sensors.read.completed");
            this.metricsReadInflight = registry.lazyGaugeInt64("sensors.read.inFlight",
                    () -> metricsReadStarted.get() - metricsReadCompleted.get());
            this.metricsReadElapsedTime = registry.histogramRate("sensors.read.elapsedTimeMillis", Histograms.exponential(16, 2.0, 1));
            this.metricsReadOffsetMinutes = registry.histogramRate("sensors.read.offsetTimeMinutes", Histograms.explicit(READ_OFFSET_TIME_MINUTES_BOUNDS));
        }

        public String getShardId() {
            return shardId;
        }

        @Override
        public int estimateCount() {
            return registry.estimateCount();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
        }

        public ShardMetrics combine(ShardMetrics right) {
            this.metricsReadStarted.combine(right.metricsReadStarted);
            this.metricsReadCompleted.combine(right.metricsReadCompleted);
            this.metricsReadElapsedTime.combine(right.metricsReadElapsedTime);
            this.metricsReadOffsetMinutes.combine(right.metricsReadOffsetMinutes);
            return this;
        }

        public long started(int countMetrics) {
            metricsReadStarted.add(countMetrics);
            return System.nanoTime();
        }

        public void addReadInterval(Interval interval) {
            addReadInterval(1, interval);
        }

        public void addReadInterval(int count, Interval interval) {
            long now = System.currentTimeMillis();
            long offsetMillis = now - interval.getBeginMillis();
            this.metricsReadOffsetMinutes.record(TimeUnit.MILLISECONDS.toMinutes(offsetMillis), count);
        }

        public void completed(int countMetrics, long startNanoTime) {
            this.metricsReadCompleted.add(countMetrics);
            long elapsedTimeMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanoTime);
            this.metricsReadElapsedTime.record(elapsedTimeMillis, countMetrics);
        }
    }
}
