package ru.yandex.solomon.gateway.backend.www;

import java.util.Arrays;
import java.util.LinkedHashSet;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.util.Escaper;
import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class MetricsChecksSet {

    public static final MetricsChecksSet ALL = new MetricsChecksSet(false);

    private static final char ID_DELIMITER = ';';
    private static final Escaper ID_ESCAPER = new Escaper(ID_DELIMITER + "", '\\');

    private final boolean positive;
    private final LinkedHashSet<String> metricIds;

    private MetricsChecksSet(boolean positive, LinkedHashSet<String> metricIds) {
        this.positive = positive;
        this.metricIds = metricIds;
    }

    MetricsChecksSet(boolean positive, String... metricIds) {
        this(positive, new LinkedHashSet<>(Arrays.asList(metricIds)));
    }

    public boolean include(String id) {
        return positive == metricIds.contains(id);
    }

    public boolean isPositive() {
        return positive;
    }

    public LinkedHashSet<String> getIds() {
        return metricIds;
    }

    public static String child(String parentId, String childId) {
        if (parentId.equals(childId)) {
            return parentId;
        }
        return "src:" + parentId + ":" + childId;
    }

    public static MetricsChecksSet parse(String value) {
        if (value.isEmpty()) {
            return new MetricsChecksSet(false);
        }

        char startSymbol = value.charAt(0);
        String tail = value.substring(1);

        boolean positive;
        switch (startSymbol) {
            case '+':
                positive = true;
                break;
            case '-':
                positive = false;
                break;
            default:
                throw new RuntimeException("failed to parse checks: " + value);
        }

        LinkedHashSet<String> metricIds = parseMetricIdsFromTail(tail, positive);
        return new MetricsChecksSet(positive, metricIds);
    }

    private static LinkedHashSet<String> parseMetricIdsFromTail(String tail, boolean positive) {
        LinkedHashSet<String> metricIds = new LinkedHashSet<>();

        if (!tail.isEmpty()) {
            ParserSupport parser = new ParserSupport(tail);

            while (parser.hasNext()) {
                String metricId = ID_ESCAPER.consumeUntil(parser, ID_DELIMITER);
                if (parser.hasNext()) {
                    parser.consume(ID_DELIMITER);
                }
                metricIds.add(metricId);
                if (positive && metricId.startsWith("src:")) {
                    metricIds.add(metricId.substring("src:".length(), metricId.lastIndexOf(":")));
                }
            }
        }
        return metricIds;
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof MetricsChecksSet)) {
            return false;
        }
        MetricsChecksSet other = (MetricsChecksSet) o;
        return this.positive == other.positive && this.metricIds.equals(other.metricIds);
    }

    @Override
    public int hashCode() {
        return Boolean.hashCode(positive) * 31 + metricIds.hashCode();
    }

    @Override
    public String toString() {
        return (positive ? "+" : "-") + metricIds;
    }
}
