package ru.yandex.solomon.gateway.data;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.metrics.client.combined.OldModeResult;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * Data client metrics to measure common metrics in each project for each client methods
 *
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DataClientMetrics implements MetricSupplier {

    private final MetricRegistry metricRegistry;
    private final ConcurrentHashMap<String, ProjectMetrics> metricsByProjectId;
    private final Rate truncatedResponsesTotal;
    private final Rate summaryResponsesTotal;
    private final Rate normalResponsesTotal;

    public DataClientMetrics() {
        this.metricRegistry = new MetricRegistry();
        this.metricsByProjectId = new ConcurrentHashMap<>();
        this.truncatedResponsesTotal = metricRegistry.rate("dataClient.oldMode.truncatedResponses.total");
        this.summaryResponsesTotal = metricRegistry.rate("dataClient.oldMode.summaryResponses.total");
        this.normalResponsesTotal = metricRegistry.rate("dataClient.oldMode.normalResponses.total");
    }

    public void register(OldModeResult result) {
        if (result.equals(OldModeResult.DEFAULT)) {
            this.normalResponsesTotal.inc();
        } else {
            if (result.isTruncated()) {
                this.truncatedResponsesTotal.inc();
            }
            if (result.isSummary()) {
                this.summaryResponsesTotal.inc();
            }
        }
    }

    ProjectMetrics getProjectMetrics(String projectId) {
        var result = metricsByProjectId.get(projectId);
        if (result != null) {
            return result;
        }

        return metricsByProjectId.computeIfAbsent(projectId, ProjectMetrics::new);
    }

    @Override
    public int estimateCount() {
        return (metricsByProjectId.size() + 1) * 5;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        metricRegistry.append(tsMillis, commonLabels, consumer);
        var total = new ProjectMetrics("total");
        for (ProjectMetrics projectMetrics : metricsByProjectId.values()) {
            projectMetrics.append(tsMillis, commonLabels, consumer);
            total.combine(projectMetrics);
        }
        total.append(tsMillis, commonLabels, consumer);
    }

    public static class ProjectMetrics {
        private final MetricRegistry registry;
        private final AsyncMetrics requests;

        ProjectMetrics(String projectId) {
            this.registry = new MetricRegistry(Labels.of("projectId", projectId));
            this.requests = new AsyncMetrics(registry, "dataClient.call");
        }

        void callStarted() {
            requests.callStarted();
        }

        void callCompleted(long durationNanos) {
            requests.callCompletedOk(TimeUnit.NANOSECONDS.toMillis(durationNanos));
        }

        void callFailed(long durationNanos) {
            requests.callCompletedError(TimeUnit.NANOSECONDS.toMillis(durationNanos));
        }

        void combine(ProjectMetrics other) {
            this.requests.combine(other.requests);
        }

        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
        }
    }
}
