package ru.yandex.solomon.gateway.data;

import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import org.apache.commons.math3.util.ArithmeticUtils;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.core.conf.PushOrPull;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.exceptions.ServiceUnavailableException;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
public class DataClientRequestCustomizer {
    // It's neccessary to support special behavior for pull and push shards both:
    // - pull - downsample timeseries that contain correctly marked stepMillis with fillOption=NULL
    // - push - ignore incorrectly marked stepMillis in timeseries and downsample with fillOption=NONE
    // - both - min gridMillis must be >= service.stepSeconds aka service interval
    private final SolomonConfHolder confHolder;
    private final FeatureFlagsHolder featureFlagsHolder;

    public DataClientRequestCustomizer(SolomonConfHolder confHolder, FeatureFlagsHolder featureFlagsHolder) {
        this.confHolder = confHolder;
        this.featureFlagsHolder = featureFlagsHolder;
    }

    public DownsamplingOptions customizeDownsamplingOpts(DownsamplingOptions downsamplingOpts, Interval interval, List<Selectors> selectors, long minBeginMillisOfData) {
        // First try to deduce the shardGridMillis
        if (selectors.isEmpty()) {
            return downsamplingOpts;
        }

        ShardOpts shardOpts = mergeShardOpts(resolveShardOpts(selectors, minBeginMillisOfData, Instant.now().toEpochMilli()));
        if (shardOpts == null || shardOpts.gridSec <= 0) {
            return downsamplingOpts;
        }

        long shardGridMillis = TimeUnit.SECONDS.toMillis(shardOpts.gridSec);
        var builder = downsamplingOpts.toBuilder()
                .setShardGridMillis(shardGridMillis);

        // For downsampling=off and downsampling=by_interval we're done
        if (downsamplingOpts.getDownsamplingType() != DownsamplingType.BY_POINTS) {
            return builder.build();
        }

        // The following logic converts by_points to by_interval
        long autoGridMillis = downsamplingOpts.computeGridMillis(interval.duration().toMillis());
        // This cratch is about to support min grid millis from service interval
        // It fixes problems in new Data API without this cratch: SOLOMON-5011 and SOLOMON-4297
        if (autoGridMillis < shardGridMillis) {
            autoGridMillis = shardGridMillis;
        }

        builder.setDownsamplingType(DownsamplingType.BY_INTERVAL);

        if (shardOpts.hasGridFlag) {
            // We should use rounding up here as required by the "maxPoints" logic.
            // E.g.: interval=30m, shardGridMillis=15s, maxPoints=100.
            // Expected gridMillis will be equal to 30s (60 points).
            return builder
                    .setGridMillis(InstantUtils.ceil(autoGridMillis, shardGridMillis))
                    .build();
        }

        builder.setGridMillis(autoGridMillis);

        // This cratch is about to support no gap filling for push shard metrics by default.
        // History time:
        // Previously, all push metrics were written with stepMillis=0 parameter and stepMillis=0 was been treated as "don't fill gaps between this and next point".
        // Then we made the following changes in aggregation in Coremon and break this logic for push metrics: https://a.yandex-team.ru/arc/commit/3255738
        // This was fixed by Gateway side the following commit: https://a.yandex-team.ru/arc/commit/3258545
        // Then we implement downsampling operations in Stockpile and drop stepMillis=0 logic: https://a.yandex-team.ru/arc/commit/3804623
        if (shardOpts.isPushShard && !downsamplingOpts.isIgnoreMinStepMillis()) {
            builder.setDownsamplingFill(OperationDownsampling.FillOption.NONE);
        }

        return builder.build();
    }

    private Set<ShardOpts> resolveShardOpts(List<Selectors> selectors, long beginMillis, long nowMillis) {
        var conf = confHolder.getConf();
        if (conf == null) {
            throw new ServiceUnavailableException("Shard config not loaded yet");
        }

        Set<ShardOpts> result = new HashSet<>(selectors.size());
        for (var selector : selectors) {
            ShardKey shardKey = ShardSelectors.getShardKeyOrNull(selector);
            if (shardKey != null) {
                var shard = conf.findShardOrNull(shardKey);
                if (shard != null) {
                    result.add(createOpts(shard, beginMillis, nowMillis));
                }
                continue;
            }

            var shardOnlySelectors = ShardSelectors.onlyShardKey(selector);
            if (shardOnlySelectors.isEmpty()) {
                continue;
            }

            conf.getCorrectShardsStream()
                    .filter(shard -> shardOnlySelectors.match(shard.shardKey()))
                    .map(shard -> this.createOpts(shard, beginMillis, nowMillis))
                    .forEach(result::add);
        }

        return result;
    }

    @Nullable
    private ShardOpts mergeShardOpts(Set<ShardOpts> opts) {
        return opts.stream().reduce(this::mergeShardOpts).orElse(null);
    }

    private ShardOpts createOpts(ShardConfDetailed shard, long beginMillis, long nowMillis) {
        ShardOpts result = new ShardOpts();
        result.hasGridFlag = featureFlagsHolder.hasFlag(FeatureFlag.GRID_DOWNSAMPLING, shard.getNumId());
        result.isPushShard = shard.getPushOrPull() == PushOrPull.PUSH;
        var policy = DecimPoliciesPredefined.policyFromProto(shard.getDecimPolicy().toProto());
        var item = policy.findOldest(beginMillis, nowMillis);
        var gridPolicySeconds = item == null ? 0 : Math.toIntExact(TimeUnit.MILLISECONDS.toSeconds(item.getStepMillis()));
        result.gridSec = Math.max(shard.getGridSec(), gridPolicySeconds);
        return result;
    }

    private ShardOpts mergeShardOpts(ShardOpts left, ShardOpts right) {
        ShardOpts result = new ShardOpts();
        result.isPushShard = left.isPushShard && right.isPushShard;
        result.hasGridFlag = left.hasGridFlag && right.hasGridFlag;
        if (left.gridSec == right.gridSec) {
            result.gridSec = left.gridSec;
        } else if (left.gridSec == Service.GRID_ABSENT || right.gridSec == Service.GRID_ABSENT) {
            result.gridSec = Service.GRID_ABSENT;
        } else {
            result.gridSec = ArithmeticUtils.lcm(left.gridSec, right.gridSec);
        }
        return result;
    }

    private static class ShardOpts extends DefaultObject {
        private boolean isPushShard; // affects how we fill no data
        private int gridSec; //
        private boolean hasGridFlag; // change points and step depends on max points, interval, shardGridSec

        public ShardOpts() {
        }
    }
}
