package ru.yandex.solomon.gateway.data;

import java.time.Instant;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DataRequest {
    private final String projectId;
    private final String clusterName;
    private final String program;
    private final DownsamplingOptions downsampling;
    private final Interval interval;

    private final String forceCluster;
    private final boolean useNewFormat;
    private final boolean oldMode;
    private final boolean summaryOnly;
    private final RequestProducer producer;
    private final String subjectId;
    private final DeprOpts deprOpts;
    private final Instant deadline;
    @Nullable
    private final SelVersion version;

    private DataRequest(Builder builder) {
        this.projectId = builder.projectId;
        this.clusterName = builder.clusterName;
        this.program = builder.program;
        this.downsampling = builder.downsampling;
        this.interval = builder.interval;
        this.forceCluster = builder.forceCluster;
        this.useNewFormat = builder.useNewFormat;
        this.oldMode = builder.oldMode;
        this.summaryOnly = builder.summaryOnly;
        this.producer = builder.producer;
        this.deprOpts = builder.deprOpts;
        this.deadline = builder.deadline;
        this.version = builder.version;
        this.subjectId = builder.subjectId;
    }

    public Builder toBuilder() {
        Builder builder = new Builder();
        builder.projectId = this.projectId;
        builder.clusterName = this.clusterName;
        builder.program = this.program;
        builder.downsampling = this.downsampling;
        builder.interval = this.interval;
        builder.forceCluster = this.forceCluster;
        builder.useNewFormat = this.useNewFormat;
        builder.summaryOnly = this.summaryOnly;
        builder.oldMode = this.oldMode;
        builder.producer = this.producer;
        builder.deprOpts = this.deprOpts;
        builder.deadline = this.deadline;
        builder.version = this.version;
        return builder;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getProjectId() {
        return projectId;
    }

    public String getClusterName() {
        return clusterName;
    }

    public String getProgram() {
        return program;
    }

    public DownsamplingOptions getDownsampling() {
        return downsampling;
    }

    public Interval getInterval() {
        return interval;
    }

    public String getForceCluster() {
        return forceCluster;
    }

    public boolean isUseNewFormat() {
        return useNewFormat;
    }

    public boolean isOldMode() {
        return oldMode;
    }

    public boolean isSummaryOnly() {
        return summaryOnly;
    }

    public RequestProducer getProducer() {
        return producer;
    }

    public String getSubjectId() {
        return subjectId;
    }

    public DeprOpts getDeprOpts() {
        return deprOpts;
    }

    public Instant getDeadline() {
        return deadline;
    }

    @Nullable
    public SelVersion getVersion() {
        return version;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DataRequest that = (DataRequest) o;
        return useNewFormat == that.useNewFormat &&
                oldMode == that.oldMode &&
                summaryOnly == that.summaryOnly &&
                projectId.equals(that.projectId) &&
                clusterName.equals(that.clusterName) &&
                program.equals(that.program) &&
                downsampling.equals(that.downsampling) &&
                interval.equals(that.interval) &&
                forceCluster.equals(that.forceCluster) &&
                producer == that.producer &&
                subjectId.equals(that.subjectId) &&
                deprOpts == that.deprOpts &&
                deadline.equals(that.deadline) &&
                version == that.version;
    }

    @Override
    public int hashCode() {
        return Objects.hash(projectId, clusterName, program, downsampling, interval, forceCluster, useNewFormat,
                oldMode, summaryOnly, producer, subjectId, deprOpts, deadline, version);
    }

    @Override
    public String toString() {
        return "DataRequest{" +
                "projectId='" + projectId + '\'' +
                ", clusterName='" + clusterName + '\'' +
                ", program='" + program + '\'' +
                ", downsampling=" + downsampling +
                ", interval=" + interval +
                ", forceCluster='" + forceCluster + '\'' +
                ", useNewFormat=" + useNewFormat +
                ", oldMode=" + oldMode +
                ", summaryOnly=" + summaryOnly +
                ", producer=" + producer +
                ", subjectId=" + subjectId +
                ", deprOpts=" + deprOpts +
                ", deadline=" + deadline +
                ", version=" + version +
                '}';
    }

    public static class Builder {
        private String projectId = "";
        private String clusterName = "";
        private String program = "";
        private DownsamplingOptions downsampling = DownsamplingOptions.newBuilder().build();
        private Interval interval = Interval.EMPTY;
        private String forceCluster = "";
        private boolean useNewFormat;
        private boolean oldMode;
        private boolean summaryOnly = false;
        private RequestProducer producer = RequestProducer.REQUEST_PRODUCER_UNSPECIFIED;
        private String subjectId;
        private DeprOpts deprOpts = DeprOpts.DATA_API;
        private Instant deadline = Instant.EPOCH;
        @Nullable
        private SelVersion version = null;

        public Builder setProjectId(String projectId) {
            this.projectId = projectId;
            return this;
        }

        public Builder setClusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public Builder setProgram(String program) {
            this.program = program;
            return this;
        }

        public Builder setDownsampling(@Nullable DownsamplingOptions downsampling) {
            this.downsampling = downsampling;
            return this;
        }

        public Builder setInterval(@Nullable Interval interval) {
            this.interval = interval;
            return this;
        }

        public Builder setForceCluster(String forceCluster) {
            this.forceCluster = forceCluster;
            return this;
        }

        public Builder setUseNewFormat(boolean useNewFormat) {
            this.useNewFormat = useNewFormat;
            return this;
        }

        public Builder setOldMode(boolean oldMode) {
            this.oldMode = oldMode;
            return this;
        }

        public Builder setSummaryOnly(boolean summaryOnly) {
            this.summaryOnly = summaryOnly;
            return this;
        }

        public Builder setProducer(RequestProducer producer) {
            this.producer = producer;
            return this;
        }

        public Builder setDeprOpts(DeprOpts deprOpts) {
            this.deprOpts = deprOpts;
            return this;
        }

        public Builder setDeadline(Instant deadline) {
            this.deadline = deadline;
            return this;
        }

        public Builder setVersion(SelVersion version) {
            this.version = version;
            return this;
        }

        public Builder setSubjectId(String subjectId) {
            this.subjectId = subjectId;
            return this;
        }

        public DataRequest build() {
            return new DataRequest(this);
        }
    }
}
