package ru.yandex.solomon.gateway.data;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.combined.DataLimits;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DownsamplingOptions {

    private final DownsamplingType downsamplingType;
    private final Aggregation downsamplingAggr;
    private final OperationDownsampling.FillOption downsamplingFill;
    private final long gridMillis;
    private final long shardGridMillis;
    private final int points;
    private final boolean ignoreMinStepMillis;

    private DownsamplingOptions(Builder builder) {
        downsamplingType = builder.downsamplingType;
        downsamplingAggr = builder.downsamplingAggr;
        downsamplingFill = builder.downsamplingFill;
        gridMillis = builder.gridMillis;
        shardGridMillis = builder.shardGridMillis;
        points = builder.points;
        ignoreMinStepMillis = builder.ignoreMinStepMillis;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public DownsamplingType getDownsamplingType() {
        return downsamplingType;
    }

    public Aggregation getDownsamplingAggr() {
        return downsamplingAggr;
    }

    public OperationDownsampling.FillOption getDownsamplingFill() {
        return downsamplingFill;
    }

    public long getGridMillis() {
        return gridMillis;
    }

    public long getShardGridMillis() {
        return shardGridMillis;
    }

    public int getPoints() {
        return points;
    }

    public boolean isIgnoreMinStepMillis() {
        return ignoreMinStepMillis;
    }

    public long computeGridMillis(long intervalMillis) {
        final long gridRoundMillis = TimeUnit.SECONDS.toMillis(1);

        final long gridMillis = switch (downsamplingType) {
            case OFF -> 0;
            case BY_INTERVAL -> this.gridMillis;
            case BY_POINTS -> {
                if (points <= 0) {
                    yield 0;
                }
                if (points == 1) {
                    yield intervalMillis * 2;
                }
                long stepMillis = Math.max(intervalMillis / (points - 1), gridRoundMillis);
                yield (stepMillis / gridRoundMillis) * gridRoundMillis;
            }
        };

        if (gridMillis == 0) {
            if (intervalMillis > DataLimits.MAX_INTERVAL_MILLIS_FOR_RAW_DATA) {
                String message = "interval cannot be greater than "
                    + DataLimits.MAX_INTERVAL_DAYS_FOR_RAW_DATA + " days for raw data";

                throw new BadRequestException(message);
            }

            return 0;
        }

        long pointsAfterGridMillis = intervalMillis / gridMillis;
        if (pointsAfterGridMillis > DataLimits.MAX_POINTS_COUNT) {
            throw new BadRequestException(
                "too many points: " + pointsAfterGridMillis + ", please increase gridMillis");
        }

        return gridMillis;
    }

    @Override
    public String toString() {
        return "DownsamplingOptions{" +
                "downsamplingType=" + downsamplingType +
                ", downsamplingAggr=" + downsamplingAggr +
                ", downsamplingFill=" + downsamplingFill +
                ", gridMillis=" + gridMillis +
                ", shardGridMillis=" + shardGridMillis +
                ", points=" + points +
                ", ignoreMinStepMillis=" + ignoreMinStepMillis +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DownsamplingOptions that = (DownsamplingOptions) o;
        return gridMillis == that.gridMillis &&
                points == that.points &&
                ignoreMinStepMillis == that.ignoreMinStepMillis &&
                downsamplingType == that.downsamplingType &&
                downsamplingAggr == that.downsamplingAggr &&
                downsamplingFill == that.downsamplingFill &&
                shardGridMillis == that.shardGridMillis;
    }

    @Override
    public int hashCode() {
        return Objects.hash(downsamplingType, downsamplingAggr, downsamplingFill, gridMillis, points, ignoreMinStepMillis, shardGridMillis);
    }

    @Nullable
    public OperationDownsampling toOperation(long intervalMillis) {
        long gridMillis = computeGridMillis(intervalMillis);
        if (gridMillis == 0) {
            return null;
        }
        return OperationDownsampling.newBuilder()
                .setGridMillis(gridMillis)
                .setAggregation(getDownsamplingAggr())
                .setFillOption(getDownsamplingFill())
                .setIgnoreMinStepMillis(isIgnoreMinStepMillis())
                .build();
    }

    public static final class Builder {
        private DownsamplingType downsamplingType = DownsamplingType.BY_POINTS;
        private Aggregation downsamplingAggr = Aggregation.DEFAULT_AGGREGATION;
        private OperationDownsampling.FillOption downsamplingFill =
            OperationDownsampling.FillOption.NULL;
        private long gridMillis = 0;
        private long shardGridMillis = 0;
        private int points = 500;
        private boolean ignoreMinStepMillis = false;

        private Builder() {
        }

        private Builder(DownsamplingOptions opts) {
            this.downsamplingType = opts.downsamplingType;
            this.downsamplingAggr = opts.downsamplingAggr;
            this.downsamplingFill = opts.downsamplingFill;
            this.gridMillis = opts.gridMillis;
            this.shardGridMillis = opts.shardGridMillis;
            this.points = opts.points;
            this.ignoreMinStepMillis = opts.ignoreMinStepMillis;
        }

        public Builder setDownsamplingType(DownsamplingType downsamplingType) {
            this.downsamplingType = downsamplingType;
            return this;
        }

        public Builder setDownsamplingAggr(Aggregation downsamplingAggr) {
            this.downsamplingAggr = downsamplingAggr;
            return this;
        }

        public Builder setDownsamplingFill(OperationDownsampling.FillOption downsamplingFill) {
            this.downsamplingFill = downsamplingFill;
            return this;
        }

        public Builder setGridMillis(long gridMillis) {
            this.gridMillis = gridMillis;
            return this;
        }

        public Builder setShardGridMillis(long shardGridMillis) {
            this.shardGridMillis = shardGridMillis;
            return this;
        }

        public Builder setPoints(int points) {
            this.points = points;
            return this;
        }

        public Builder setIgnoreMinStepMillis(boolean ignoreMinStepMillis) {
            this.ignoreMinStepMillis = ignoreMinStepMillis;
            return this;
        }

        public DownsamplingOptions build() {
            return new DownsamplingOptions(this);
        }
    }
}
