package ru.yandex.solomon.gateway.data;

import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.metrics.client.AbstractRequest;
import ru.yandex.solomon.metrics.client.MetricsClientMetrics;

/**
 * @author Alexey Trushkin
 */
public class MetricsClientSubjectMetrics implements MetricsClientMetrics {

    private final MetricRegistry registry;
    private final ConcurrentMap<String, InnerMetrics> metrics;
    private final InnerMetrics total;

    public MetricsClientSubjectMetrics() {
        registry = new MetricRegistry();
        metrics = new ConcurrentHashMap<>();
        total = metricsBySubject("total");
    }

    @Override
    public void hitMetricsRead(AbstractRequest request, int readBytes, int readPointCount) {
        metricsBySubject(request.getSubjectId()).hit(readBytes, readPointCount);
        total.hit(readBytes, readPointCount);
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }

    public Collection<SubjectInfo> getSubjectsInfo() {
        return metrics.values().stream()
                .map(this::toRecord)
                .collect(Collectors.toList());
    }

    public SubjectInfo toRecord(InnerMetrics metrics) {
        return new SubjectInfo(metrics.subjectId,
                metrics.readBytesMeter.getRate(TimeUnit.SECONDS),
                metrics.readPointsMeter.getRate(TimeUnit.SECONDS),
                metrics.readTimeseriesMeter.getRate(TimeUnit.SECONDS));
    }

    private InnerMetrics metricsBySubject(String subjectId) {
        return metrics.computeIfAbsent(subjectId, InnerMetrics::new);
    }

    private class InnerMetrics {

        private final Rate readBytesRate;
        private final Meter readBytesMeter;
        private final Rate readPointsRate;
        private final Meter readPointsMeter;
        private final Rate readTimeseriesRate;
        private final Meter readTimeseriesMeter;
        private final String subjectId;

        private InnerMetrics(String subjectId) {
            var commonLabels = Labels.of("subject_id", subjectId, "host", "cluster");
            this.subjectId = subjectId;
            readBytesRate = registry.rate("subject.read_bytes", commonLabels);
            readPointsRate = registry.rate("subject.read_points", commonLabels);
            readTimeseriesRate = registry.rate("subject.read_timeseries", commonLabels);
            readBytesMeter = Meter.of(ExpMovingAverage.fiveMinutes());
            readPointsMeter = Meter.of(ExpMovingAverage.fiveMinutes());
            readTimeseriesMeter = Meter.of(ExpMovingAverage.fiveMinutes());
        }

        public void hit(int readBytes, int readPointCount) {
            readBytesRate.add(readBytes);
            readBytesMeter.mark(readBytes);

            readPointsRate.add(readPointCount);
            readPointsMeter.mark(readPointCount);

            readTimeseriesRate.inc();
            readTimeseriesMeter.mark();
        }
    }

    public record SubjectInfo(String subjectId, double readBytesRate, double readPointsRate,
                              double readTimeseriesRate) {
    }
}
