package ru.yandex.solomon.gateway.data;

import java.util.Arrays;
import java.util.Random;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class RandomSeriesGenerator extends AggrGraphDataListIterator {

    private static final long[] PERIODS = new long[] {
            TimeUnit.SECONDS.toMillis(5),
            TimeUnit.SECONDS.toMillis(20),
            TimeUnit.MINUTES.toMillis(1),
            TimeUnit.MINUTES.toMillis(5),
            TimeUnit.MINUTES.toMillis(30),
            TimeUnit.HOURS.toMillis(3),
            TimeUnit.DAYS.toMillis(1),
            TimeUnit.DAYS.toMillis(7)
    };

    private static final double[] FREQUENCIES;
    private static final double[] AMPLITUDE_SCALE;

    static {
        FREQUENCIES = Arrays.stream(PERIODS).mapToDouble(ms -> 2d * Math.PI / ms).toArray();
        AMPLITUDE_SCALE = Arrays.stream(PERIODS).mapToDouble(ms -> Math.log(ms * 1e-3)).toArray();
    }

    private final Random random;
    private final long toMillis;
    private final long gridMillis;
    private final int count;
    private long ts;

    private final double[] amplitudes;
    private final double[] phases;

    public RandomSeriesGenerator(String key, long fromMillis, long toMillis, long gridMillis) {
        super(StockpileColumns.minColumnSet(MetricType.DGAUGE));

        if (gridMillis <= 0) {
            throw new IllegalArgumentException("gridMillis must be positive");
        }

        this.random = new Random(key.hashCode());
        this.toMillis = toMillis;
        this.gridMillis = gridMillis;
        this.ts = fromMillis;
        this.count = Math.toIntExact((toMillis - fromMillis + gridMillis - 1) / gridMillis);

        this.amplitudes = Arrays.stream(AMPLITUDE_SCALE).map(scale -> random.nextGaussian() * scale).toArray();
        this.phases = Arrays.stream(AMPLITUDE_SCALE).map(ignore -> 2 * Math.PI * random.nextDouble()).toArray();
    }

    @Override
    public int estimatePointsCount() {
        return count;
    }

    @Override
    public boolean next(AggrPoint target) {
        if (!(ts < toMillis)) {
            return false;
        }

        target.setTsMillis(ts);
        target.setValue(genValue(ts));

        ts += gridMillis;

        return true;
    }

    private double genValue(double ts) {
        double s = 0;
        for (int i = 0; i < amplitudes.length; i++) {
            s += amplitudes[i] * Math.cos(FREQUENCIES[i] * ts + phases[i]);
        }
        return s;
    }
}
