package ru.yandex.solomon.gateway.data;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.config.protobuf.frontend.ShardMetricsCurrentValueConfig;
import ru.yandex.solomon.core.conf.ShardMetricsQuotaReader;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Alexey Trushkin
 */
public class ShardMetricsQuotaReaderImpl implements ShardMetricsQuotaReader {
    private static final Logger logger = LoggerFactory.getLogger(ShardMetricsQuotaReaderImpl.class);
    private static final String DATA_READ_TEMPLATE = "avg(drop_tail(series_max({%s, projectId='%s', shardId='%s' }), 1))";
    private final DataClient dataClient;
    private final ShardMetricsCurrentValueConfig configs;

    public ShardMetricsQuotaReaderImpl(DataClient dataClient, ShardMetricsCurrentValueConfig configs) {
        this.dataClient = dataClient;
        this.configs = configs;
    }

    @Override
    public CompletableFuture<CurrentQuota> readCurrentMetricsValue(String id, String projectId) {
        if (configs == null) {
            return CompletableFuture.completedFuture(CurrentQuota.EMPTY);
        }
        return CompletableFutures.allOf(List.of(
                readValue(configs.getMetricsPerUrlSelectorPrefix(), id, projectId),
                readValue(configs.getFileMetricsSelectorPrefix(), id, projectId),
                readValue(configs.getMemMetricsSelectorPrefix(), id, projectId),
                readValue(configs.getFetcherResponseSelectorPrefix(), id, projectId)
        )).thenApply(doubles -> new CurrentQuota(doubles.get(0), doubles.get(1), doubles.get(2), doubles.get(3)));
    }

    private CompletableFuture<Double> readValue(String prefix, String shardId, String projectId) {
        return dataClient.readData(DataRequest.newBuilder()
                .setProjectId(configs.getProject())
                .setProgram(String.format(DATA_READ_TEMPLATE, prefix, projectId, shardId))
                .setInterval(Interval.before(Instant.now(), Duration.ofMinutes(5)))
                .setDownsampling(DownsamplingOptions.newBuilder().setDownsamplingType(DownsamplingType.BY_POINTS).setPoints(10).build())
                .build())
                .thenApply(dataResponse -> {
                    var val = dataResponse.getEvalResult().castToScalar().getValue();
                    if (Double.isNaN(val)) {
                        return 0.0;
                    }
                    return val;
                })
                .handle((o, throwable) -> {
                    if (throwable != null) {
                        logger.error("Cant read " + prefix + " from " + shardId, throwable);
                        return 0.0;
                    }
                    return o;
                });
    }
}
