package ru.yandex.solomon.gateway.data;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadRequest;
import ru.yandex.solomon.metrics.client.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.ResolveOneWithNameRequest;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.MetricKeyPartial;

import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureMetabaseStatusValid;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureStockpileStatusValid;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class StockpileMetricClient {

    private final MetricsClient metricsClient;
    private final ClusterMapper clusterMapper;

    public StockpileMetricClient(
        MetricsClient metricsClient,
        ClusterMapper clusterMapper)
    {
        this.metricsClient = metricsClient;
        this.clusterMapper = clusterMapper;
    }

    public CompletableFuture<StockpileMetricDataResponse> read(
        String projectId,
        StockpileMetricDataRequest request,
        Instant deadline,
        String subjectId)
    {
        String destination = clusterMapper.byParamOrNull(request.getForceCluster());

        Labels labelsWithProject = request.getLabels().add(LabelKeys.PROJECT, projectId);

        CompletableFuture<MetricKey> future;

        if (request.getName().isEmpty()) {
            ResolveOneRequest metabaseRequest = ResolveOneRequest.newBuilder()
                .setLabels(labelsWithProject)
                .setDestination(destination)
                .setDeadline(deadline)
                .build();

            future = metricsClient.resolveOne(metabaseRequest).thenApply(metabaseResponse -> {
                ensureMetabaseStatusValid(metabaseResponse.getStatus());
                return metabaseResponse.getMetric();
            });
        } else {
            ResolveOneWithNameRequest metabaseRequest = ResolveOneWithNameRequest.newBuilder()
                .setMetric(new MetricKeyPartial(request.getName(), labelsWithProject))
                .setDestination(destination)
                .setDeadline(deadline)
                .build();

            future = metricsClient.resolveOneWithName(metabaseRequest).thenApply(metabaseResponse -> {
                ensureMetabaseStatusValid(metabaseResponse.getStatus());
                return metabaseResponse.getMetric();
            });
        }

        return future.thenCompose(metricKey -> {
            List<ExtendedStockpileKey> extendedStockpileKeys =
                metricKey.getStockpileKeys().stream()
                    .map(key -> {
                        String host = metricsClient
                            .getStockpileHostForShardId(key.getDestination(), key.getShardId());
                        return new ExtendedStockpileKey(key, host);
                    })
                    .collect(Collectors.toList());

            ReadRequest readRequest = ReadRequest.newBuilder()
                .setKey(metricKey)
                .setInterval(request.getInterval())
                .setDeadline(deadline)
                .setDestination(destination)
                .setSubjectId(subjectId)
                .build();

            return metricsClient.read(readRequest).thenApply(stockpileResponse -> {
                ensureStockpileStatusValid(stockpileResponse.getStatus());

                return new StockpileMetricDataResponse(
                    metricKey,
                    extendedStockpileKeys,
                    stockpileResponse.getSource()
                );
            });
        });
    }
}
