package ru.yandex.solomon.gateway.data.exceptions;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.reflect.ClassPath;

import ru.yandex.solomon.expression.exceptions.SelException;
import ru.yandex.solomon.metrics.client.exceptions.DataClientException;

/**
 * Registry for exception classes that provide machine-readable details.
 * This class provides up-to-date information about the fields.
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class DataClientExceptionRegistry {
    public static final Map<String, ObjectNode> PER_TYPE_SCHEMA;

    static {
        try {
            HashMap<String, ObjectNode> tmp = new HashMap<>();

            tmp.putAll(scanForDataClientExceptions());
            tmp.putAll(scanForSelExceptions());

            PER_TYPE_SCHEMA = tmp;
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static Map<String, ObjectNode> scanForSelExceptions() throws IOException {
        ClassPath cp = ClassPath.from(DataClientException.class.getClassLoader());
        String packageName = DataClientException.class.getPackageName();
        Set<ClassPath.ClassInfo> detailedExceptions = cp.getTopLevelClasses(packageName);
        return detailedExceptions.stream()
                .map(ClassPath.ClassInfo::load)
                .filter(clazz -> DataClientException.class != clazz)
                .filter(clazz -> DataClientException.class.isAssignableFrom(clazz))
                .map(clazz -> (Class<? extends DataClientException>) clazz)
                .map(DataClientException::getExceptionSchema)
                .collect(Collectors.toUnmodifiableMap(
                        DataClientException.ExceptionSchema::getType,
                        DataClientException.ExceptionSchema::getSchema
                ));
    }

    private static Map<String, ObjectNode> scanForDataClientExceptions() throws IOException {
        ClassPath cp = ClassPath.from(SelException.class.getClassLoader());
        String packageName = SelException.class.getPackageName();
        Set<ClassPath.ClassInfo> detailedExceptions = cp.getTopLevelClasses(packageName);
        return detailedExceptions.stream()
                .map(ClassPath.ClassInfo::load)
                .filter(clazz -> SelException.class != clazz)
                .filter(clazz -> SelException.class.isAssignableFrom(clazz))
                .map(clazz -> (Class<? extends SelException>) clazz)
                .map(SelException::getExceptionSchema)
                .collect(Collectors.toUnmodifiableMap(
                        SelException.ExceptionSchema::getType,
                        SelException.ExceptionSchema::getSchema
                ));
    }

}
