package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.v3.AlertEvalStatus;
import ru.yandex.monitoring.v3.AlertWidget;
import ru.yandex.monitoring.v3.TextWidget;
import ru.yandex.monitoring.v3.Widget;
import ru.yandex.solomon.gateway.utils.url.UrlUtils;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class AlertWidgetConverter {
    private static final Pattern ALERT_SINGLESTAT_PATTERN = Pattern.compile("^.*?admin/projects/(.*?)/alerts/(.*?)/singlestat.*$");
    private static final Pattern SUB_ALERT_SINGLESTAT_PATTERN = Pattern.compile("^.*?admin/projects/(.*?)/alerts/(.*?)/subAlerts/(.*?)/singlestat.*$");
    private static final Pattern SUB_ALERTS_PATTERN = Pattern.compile("^.*?admin/projects/(.*?)/alerts/(.*?)/subAlerts.*$");
    private static final Pattern ALERT_STATES_PATTERN = Pattern.compile("^.*?admin/projects/(.*?)/alertStates.*$");

    @Nullable
    public static ParsedWidgetContent convertToAlertStatesWidget(String widgetTitle, String url) {
        Map<String, String> params = UrlUtils.parseUrlQueryArgs(url);
        List<String> alertIds = arrayFromParams(params, "alertIds");
        List<AlertEvalStatus> filterByEvalStatus = filterByEvalStatusFromParams(params);

        var matcher = ALERT_STATES_PATTERN.matcher(url);
        if (matcher.matches()) {
            var projectId = matcher.group(1);

            List<DashScheme.WidgetScheme<ParsedWidgetContent>> widgets = new ArrayList<>();

            for (int i = 0; i < alertIds.size(); i++) {
                String alertId = alertIds.get(i);
                ParsedWidgetContent widget = new ParsedWidgetContent.SimpleWidget(Widget.newBuilder()
                        .setAlert(AlertWidget.newBuilder()
                                .setTitle(widgetTitle + " (" + alertId + ")")
                                .setProjectId(projectId)
                                .setAlertId(alertId)
                                .addAllEvalStatusFilters(filterByEvalStatus)
                                .build())
                        .build());

                widgets.add(new DashScheme.WidgetScheme<>(i, 0, 1, 1, widget));
            }

            DashScheme<ParsedWidgetContent> dash = new DashScheme<>(widgets);

            return new ParsedWidgetContent.ComplexWidget(dash);
        }

        return null;
    }

    public static CompletableFuture<ParsedWidgetContent> convertToAlertSinglestatWidgetAsync(
            String widgetTitle,
            String url,
            ExternalLoader externalLoader)
    {
        Map<String, String> params = UrlUtils.parseUrlQueryArgs(url);

        var widgetBuilder = AlertWidget.newBuilder();

        var matcher = ALERT_SINGLESTAT_PATTERN.matcher(url);
        if (matcher.matches()) {
            var projectId = matcher.group(1);
            var alertId = matcher.group(2);

            setCommonAlertWidgetParams(widgetBuilder, widgetTitle, projectId, alertId, params);

            var widget = Widget.newBuilder().setAlert(widgetBuilder.build()).build();

            return CompletableFuture.completedFuture(new ParsedWidgetContent.SimpleWidget(widget));
        }

        matcher = SUB_ALERT_SINGLESTAT_PATTERN.matcher(url);
        if (matcher.matches()) {
            var projectId = matcher.group(1);
            var alertId = matcher.group(2);
            var subAlertId = matcher.group(3);

            return externalLoader.loadSubAlertLabels(projectId, alertId, subAlertId)
                    .thenApply(subAlertLabels -> {
                        if (subAlertLabels == null) {
                            return new ParsedWidgetContent.SimpleWidget(Widget.newBuilder()
                                    .setText(TextWidget.newBuilder()
                                            .setText("Unknown sub alert: " + projectId + "/" + alertId + "/" + subAlertId)
                                            .build())
                                    .build());
                        }

                        setCommonAlertWidgetParams(widgetBuilder, widgetTitle, projectId, alertId, params);
                        if (!subAlertLabels.isEmpty()) {
                            var labelsFilter = Selectors.format(Selectors.of(subAlertLabels));
                            widgetBuilder.setLabelsFilter(labelsFilter);
                        }

                        var widget = Widget.newBuilder().setAlert(widgetBuilder.build()).build();

                        return new ParsedWidgetContent.SimpleWidget(widget);
                    });
        }

        return CompletableFuture.completedFuture(null);
    }

    @Nullable
    public static ParsedWidgetContent convertToSubAlertsWidget(String widgetTitle, String url) {
        var matcher = SUB_ALERTS_PATTERN.matcher(url);
        if (matcher.matches()) {
            var projectId = matcher.group(1);
            var alertId = matcher.group(2);

            Map<String, String> params = UrlUtils.parseUrlQueryArgs(url);
            var widgetBuilder = AlertWidget.newBuilder();

            setCommonAlertWidgetParams(widgetBuilder, widgetTitle, projectId, alertId, params);

            List<AlertEvalStatus> filterByEvalStatus = filterByEvalStatusFromParams(params);
            if (!filterByEvalStatus.isEmpty()) {
                widgetBuilder.addAllEvalStatusFilters(filterByEvalStatus);
            }

            String filterByLabels = params.getOrDefault("filterByLabels", "");
            if (!filterByLabels.isEmpty()) {
                widgetBuilder.setLabelsFilter(filterByLabels);
            }

            var widget = Widget.newBuilder().setAlert(widgetBuilder).build();

            return new ParsedWidgetContent.SimpleWidget(widget);
        }

        return null;
    }

    private static void setCommonAlertWidgetParams(
            AlertWidget.Builder widgetBuilder,
            String widgetTitle,
            String projectId,
            String alertId,
            Map<String, String> params)
    {
        widgetBuilder.setTitle(widgetTitle);

        List<String> annotationKeys = annotationsFromParams(params);
        if (!annotationKeys.isEmpty()) {
            widgetBuilder.addAllAnnotationKeys(annotationKeys);
        }

        widgetBuilder.setProjectId(projectId).setAlertId(alertId);
    }

    private static List<String> annotationsFromParams(Map<String, String> params) {
        List<String> annotations = arrayFromParams(params, "annotationKeys");
        annotations.remove("trafficLight.color");
        return annotations;
    }

    private static List<AlertEvalStatus> filterByEvalStatusFromParams(Map<String, String> params) {
        List<String> filterByEvalStatus = arrayFromParams(params, "filterByEvalStatus");
        return filterByEvalStatus
                .stream()
                .map(name -> {
                    try {
                        return AlertEvalStatus.valueOf("ALERT_EVAL_STATUS_" + name);
                    } catch (Exception e) {
                        return AlertEvalStatus.UNRECOGNIZED;
                    }
                })
                .filter(evalStatus -> evalStatus != AlertEvalStatus.UNRECOGNIZED)
                .collect(Collectors.toList());
    }

    private static List<String> arrayFromParams(Map<String, String> params, String paramName) {
        var annotationKeysStr = params.getOrDefault(paramName, "");

        return Arrays.stream(annotationKeysStr.split(","))
                .map(String::trim)
                .filter(str -> !str.isEmpty())
                .collect(Collectors.toList());
    }
}
