package ru.yandex.solomon.gateway.entityConverter;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.monitoring.v3.ChartWidget;
import ru.yandex.monitoring.v3.ChartWidget.NameHidingSettings;
import ru.yandex.monitoring.v3.ChartWidget.Queries;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings;
import ru.yandex.monitoring.v3.Downsampling;
import ru.yandex.monitoring.v3.Widget;
import ru.yandex.solomon.gateway.backend.page.WwwQueryArgsUtils;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.WwwBooleans;
import ru.yandex.solomon.gateway.utils.conf.GraphSettings;
import ru.yandex.solomon.gateway.utils.conf.source.GraphSettingsSourceMap;
import ru.yandex.solomon.gateway.utils.url.UrlUtils;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValues;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.selector.LabelSelectorSet;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class AutoGraphWidgetConverter {

    @Nullable
    public static ParsedWidgetContent convertToAutoGraphWidget(String widgetTitle, String url, String id) {
        ChartWidget inlineGraphWidget = createAutoGraphWidgetImpl(widgetTitle, url, id);
        if (inlineGraphWidget == null) {
            return null;
        }

        var widget = Widget.newBuilder()
                .setChart(inlineGraphWidget)
                .build();

        return new ParsedWidgetContent.SimpleWidget(widget);
    }

    @Nullable
    public static ChartWidget createAutoGraphWidgetImpl(String widgetTitle, String url, String id) {
        var queryArgs = UrlUtils.parseUrlQueryArgs(url);
        var projectParam = queryArgs.getOrDefault(LabelKeys.PROJECT, "");

        if (projectParam.isEmpty()) {
            return null;
        }

        boolean displayLegend = WwwBooleans.isTrue(queryArgs.getOrDefault(UserLinksBasic.LEGEND_QUERY_ARG, ""));

        var clusterParam = queryArgs.getOrDefault(LabelKeys.CLUSTER, LabelValues.ANY);
        var serviceParam = queryArgs.getOrDefault(LabelKeys.SERVICE, LabelValues.ANY);

        var oldSelectorsFromQueries = WwwQueryArgsUtils.selectorsFromQuery(queryArgs).toLinkedHashMap();

        var selectorsMap = new LinkedHashMap<String, String>();
        selectorsMap.put(LabelKeys.PROJECT, projectParam);
        selectorsMap.put(LabelKeys.CLUSTER, clusterParam);
        selectorsMap.put(LabelKeys.SERVICE, serviceParam);
        selectorsMap.putAll(oldSelectorsFromQueries);

        var selectors = LabelSelectorSet.fromMap(selectorsMap).toNewSelectors();

        var selectorsStr = Selectors.format(selectors);

        var source = new GraphSettingsSourceMap(queryArgs);

        String checks = queryArgs.getOrDefault(UserLinksBasic.CHECKS_QA, "");

        VisualizationSettings visualizationSettings = GraphUtils.makeVisualizationSettings(projectParam, source, null);

        Pair<Queries, Boolean> queriesAndNeedToResetChecks = makeQueries(selectorsStr, source, checks);
        Queries queries = queriesAndNeedToResetChecks.getLeft();
        boolean needToResetChecks = queriesAndNeedToResetChecks.getRight();

        NameHidingSettings nameHidingSettings = needToResetChecks
                ? NameHidingSettings.getDefaultInstance()
                : GraphUtils.makeNameHidingSettings(checks);

        return ChartWidget.newBuilder()
                .setId(id)
                .setTitle(widgetTitle)
                .setDisplayLegend(displayLegend)
                .setQueries(queries)
                .setVisualizationSettings(visualizationSettings)
                .setNameHidingSettings(nameHidingSettings)
                .build();
    }

    private static Pair<ChartWidget.Queries, Boolean> makeQueries(String selectorsStr, GraphSettingsSourceMap source, String checks) {
        boolean needToResetChecks = false;

        var expression = selectorsStr;

        var filterParam = GraphSettings.filter.getFromOrDefault(source).toLowerCase();
        var filterByParam = GraphSettings.filterBy.getFromOrDefault(source).toLowerCase();
        var filterLimitParam = GraphSettings.filterLimit.getFromOrDefault(source);
        var limitParam = GraphSettings.limit.getFromOrDefault(source);

        if (!limitParam.isEmpty()) {
            expression = "limit(" + expression + ", " + limitParam + ")";
        }

        if ("top".equals(filterParam) || "bottom".equals(filterParam)) {
            expression = filterParam + "(" + filterLimitParam + ", \"" + filterByParam + "\", " + expression + ")";
        }

        var dropNansParam = GraphSettings.dropNans.getBoolFrom(source);
        if (dropNansParam) {
            expression = "drop_nan(" + expression + ")";
        }

        var transformParam = GraphSettings.transform.getFromOrDefault(source).toLowerCase();

        switch (transformParam) {
            case "differentiate": {
                expression = "non_negative_derivative(" + expression + ")";
                break;
            }
            case "differentiate_with_negative": {
                expression = "derivative(" + expression + ")";
                break;
            }
            case "integrate": {
                expression = "integrate_fn(" + expression + ")";
                break;
            }
            case "moving_average": {
                var movingWindowParam = GraphSettings.movingWindow.getFromOrDefault(source);
                expression = "moving_avg(" + expression + ", " + movingWindowParam + ")";
                break;
            }
            case "moving_percentile": {
                var movingWindowParam = GraphSettings.movingWindow.getFromOrDefault(source);
                var movingPercentileParam = GraphSettings.movingPercentile.getFromOrDefault(source);
                expression = "moving_avg(" + expression + ", " + movingWindowParam + ", " + movingPercentileParam + ")";
                break;
            }
            case "diff": {
                expression = "diff(" + expression + ")";
                break;
            }
            case "asap":
                expression = "asap(" + expression + ")";
                break;
            default:
                break;
        }

        String finalExpression = expression;

        List<String> expressions;

        var overLinesTransform = GraphSettings.overLinesTransform.getFromOrDefault(source).toUpperCase();
        var percentilesStr = GraphSettings.percentiles.getFromOrDefault(source);
        var bucketLabel = GraphSettings.bucketLabel.getFromOrDefault(source);
        var percentiles = Arrays.stream(percentilesStr.split(","))
                .map(String::trim)
                .collect(Collectors.toList());

        switch (overLinesTransform) {
            case "PERCENTILE": {
                expressions = percentiles.stream()
                        .map(perc -> "percentile_group_lines(" + perc + ", " + finalExpression + ")")
                        .collect(Collectors.toList());
                break;
            }
            case "WEIGHTED_PERCENTILE": {
                expressions = percentiles.stream()
                        .map(perc -> "histogram_percentile(" + perc + ", " + (bucketLabel.isEmpty() ? "" : "\"" + bucketLabel + "\", ") + finalExpression + ")")
                        .collect(Collectors.toList());
                break;
            }
            case "SUMMARY": {
                var pair = QueryTransformations.makeSummary(finalExpression, checks);
                expressions = pair.getLeft();
                needToResetChecks = pair.getRight();
                break;
            }
            default: {
                expressions = List.of(finalExpression);
            }
        }

        if (GraphSettings.hideNoData.getBoolFrom(source)) {
            expressions = expressions.stream()
                    .map(expr -> "drop_empty_lines(" + expr + ")")
                    .collect(Collectors.toList());
        }

        var builder = Queries.newBuilder();

        List<Queries.Target> targets = expressions.stream()
                .map(expr -> Queries.Target.newBuilder().setQuery(expr).build())
                .collect(Collectors.toList());

        builder.addAllTargets(targets);

        Downsampling downsampling = GraphUtils.makeDownsampling(source);
        builder.setDownsampling(downsampling);

        return Pair.of(builder.build(), needToResetChecks);
    }
}
