package ru.yandex.solomon.gateway.entityConverter;

import java.util.LinkedHashMap;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.v3.ChartWidget;
import ru.yandex.monitoring.v3.TextWidget;
import ru.yandex.monitoring.v3.Widget;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.WwwBooleans;
import ru.yandex.solomon.gateway.utils.url.UrlUtils;
import ru.yandex.solomon.labels.LabelKeys;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ConfiguredGraphWidgetConverter {

    public static CompletableFuture<ParsedWidgetContent> convertToConfiguredGraphWidgetAsync(
        String widgetTitle,
        String url,
        String widgetId,
        ExternalLoader externalLoader) {
        var queryArgs = UrlUtils.parseUrlQueryArgs(url);
        var projectParam = queryArgs.getOrDefault(LabelKeys.PROJECT, "");
        var graphParam = queryArgs.getOrDefault(UserLinksBasic.GRAPH_PARAM, "").trim();

        if (projectParam.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        if (graphParam.isBlank()) {
            return CompletableFuture.completedFuture(null);
        }

        return externalLoader.loadGraph(projectParam, graphParam)
                .thenApply(graph -> {
                    if (graph == null) {
                        return new ParsedWidgetContent.SimpleWidget(Widget.newBuilder()
                                .setText(TextWidget.newBuilder()
                                        .setText("Unknown graph: " + StringInterpolator.escapePattern(projectParam) + "/" + StringInterpolator.escapePattern(graphParam))
                                        .build())
                                .build());
                    }
                    return convertToInlineGraphWidget(widgetTitle, queryArgs, graph, widgetId);
                });
    }

    private static ParsedWidgetContent convertToInlineGraphWidget(String widgetTitle, LinkedHashMap<String, String> queryArgs, Graph graph, String widgetId) {
        boolean displayLegend = WwwBooleans.isTrue(queryArgs.getOrDefault(UserLinksBasic.LEGEND_QUERY_ARG, ""));

        var graphData = GraphConverter.convertGraphData(graph, queryArgs);

        ChartWidget inlineGraph = graphData.inlineGraphWidgetBuilder
                .setId(widgetId)
                .setTitle(widgetTitle)
                .setDisplayLegend(displayLegend)
                .build();

        var widget = Widget.newBuilder().setChart(inlineGraph).build();

        return new ParsedWidgetContent.SimpleWidget(widget);
    }
}
