package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.Widget;
import ru.yandex.monitoring.v3.Widget.LayoutPosition;
import ru.yandex.solomon.config.protobuf.frontend.EntityConverterConfig;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.DashboardPanel;

@ParametersAreNonnullByDefault
public class DashConverter {
    public static final int MAX_WIDTH = 36;
    public static final int MIN_HEIGHT = 8;

    public static CompletableFuture<ru.yandex.monitoring.v3.Dashboard> convertAsync(
            Dashboard dashboard,
            ExternalLoader externalLoader,
            String widgetFilter,
            EntityConverterConfig config)
    {
        String oldDashboardUrl = config.getTargetSiteUrl() + "/admin/projects/" + dashboard.getProjectId() +
                "/dashboards/" + dashboard.getId();
        String description = ("[GENERATED from " + oldDashboardUrl + "] " + dashboard.getDescription()).trim();

        var extractedParameters = ExtractedParameters.from(dashboard.getProjectId(), dashboard.getParameters(),
                Map.of());

        var parametrization = extractedParameters.parametrization;

        return convertOldDashboardToDashSchemeAsync(dashboard, externalLoader, extractedParameters, widgetFilter, config)
                .thenApply(dashScheme -> {
                    var widgets = convertToProtoWidgets(dashScheme);

                    return ru.yandex.monitoring.v3.Dashboard.newBuilder()
                            .setId(dashboard.getGeneratedId())
                            .setProjectId(dashboard.getProjectId())
                            .setName(StringInterpolator.interpolateName(dashboard.getName()))
                            .setDescription(description)
                            .addAllWidgets(widgets)
                            .setParametrization(parametrization)
                            .setCreatedAt(Timestamps.fromMillis(dashboard.getCreatedAtMillis()))
                            .setCreatedBy(dashboard.getCreatedBy())
                            .setUpdatedAt(Timestamps.fromMillis(dashboard.getUpdatedAtMillis()))
                            .setUpdatedBy(dashboard.getUpdatedBy())
                            .build();
                });
    }

    public static CompletableFuture<DashScheme<ParsedWidgetContent>> convertOldDashboardToDashSchemeAsync(
            Dashboard dashboard,
            ExternalLoader externalLoader,
            ExtractedParameters extractedParameters,
            String widgetFilter,
            EntityConverterConfig config)
    {
        DashScheme<DashboardPanel> initialDashScheme = DashScheme.fromOldDashboard(dashboard);

        List<CompletableFuture<DashScheme.WidgetScheme<ParsedWidgetContent>>> convertedWidgetFutures = new ArrayList<>();

        List<DashScheme.WidgetScheme<DashboardPanel>> widgets = initialDashScheme.getWidgets();
        for (DashScheme.WidgetScheme<DashboardPanel> widget : widgets) {
            var widgetId = "widget_" + (widget.y + 1) + "_" + (widget.x + 1);
            if (!widgetFilter.isEmpty() && !widgetFilter.equals(widgetId)) {
                continue;
            }
            var convertedWidgetFuture = DashWidgetConverter.convertWidgetAsync(widgetId, widget.content,
                    extractedParameters.overrideParameters, externalLoader, config)
                    .thenApply(convertedWidget -> {
                        if (convertedWidget != null) {
                            return new DashScheme.WidgetScheme<>(
                                    widget.x,
                                    widget.y,
                                    widget.w,
                                    widget.h,
                                    convertedWidget
                            );
                        }
                        return null;
                    });

            convertedWidgetFutures.add(convertedWidgetFuture);
        }

        return CompletableFutures.allOf(convertedWidgetFutures).thenApply(rawConvertedWidgets -> {
            var convertedWidgets = rawConvertedWidgets.stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());

            DashScheme<ParsedWidgetContent> dash = new DashScheme<>(convertedWidgets);

            for (var widget : convertedWidgets) {
                var content = widget.content;
                if (content instanceof ParsedWidgetContent.ComplexWidget) {
                    var inlineDash = ((ParsedWidgetContent.ComplexWidget) content).dash;
                    dash = dash.replace(widget, inlineDash);
                }
            }

            dash = dash.reduceSizes();
            dash = dash.scaleByHeight(MIN_HEIGHT);
            dash = dash.scaleByFixedWidth(MAX_WIDTH);

            return dash;
        });
    }

    public static List<Widget> convertToProtoWidgets(DashScheme<ParsedWidgetContent> dash) {
        return dash.getWidgets().stream()
                .map(widget -> {
                    if (widget.content instanceof ParsedWidgetContent.SimpleWidget) {
                        var simpleWidget = ((ParsedWidgetContent.SimpleWidget) widget.content);
                        var protoWidget = simpleWidget.widget;

                        return protoWidget.toBuilder()
                                .setPosition(LayoutPosition.newBuilder()
                                        .setX(widget.x)
                                        .setY(widget.y)
                                        .setW(widget.w)
                                        .setH(widget.h)
                                        .build())
                                .build();
                    }

                    return null;
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }
}
