package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.math3.util.ArithmeticUtils;

import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.DashboardPanel;
import ru.yandex.solomon.core.db.model.DashboardRow;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class DashScheme<T> {
    private final List<WidgetScheme<T>> widgets;

    public DashScheme(List<WidgetScheme<T>> widgets) {
        this.widgets = widgets;
    }

    public List<WidgetScheme<T>> getWidgets() {
        return widgets;
    }

    public static DashScheme<DashboardPanel> fromOldDashboard(Dashboard dashboard) {
        var tableBuilder = createTableWithSpansBuilder(dashboard);
        var widgets = convertTableWithSpansBuilderToRects(tableBuilder);
        return new DashScheme<>(widgets);
    }

    private static TableWithSpansBuilder<DashboardPanel> createTableWithSpansBuilder(Dashboard dashboard) {
        DashboardRow[] rows = dashboard.getRows();
        TableWithSpansBuilder<DashboardPanel> tableBuilder = new TableWithSpansBuilder<>();
        for (int rowIdx = 0; rowIdx < rows.length; rowIdx++) {
            DashboardRow row = rows[rowIdx];
            DashboardPanel[] panels = row.getPanels();
            for (int colIdx = 0; colIdx < panels.length; colIdx++) {
                DashboardPanel panel = panels[colIdx];
                tableBuilder.setSpannedCell(rowIdx, colIdx, panel.getRowspan(), panel.getColspan(), panel);
            }
        }
        return tableBuilder;
    }

    private static List<WidgetScheme<DashboardPanel>> convertTableWithSpansBuilderToRects(TableWithSpansBuilder<DashboardPanel> tableBuilder) {
        List<WidgetScheme<DashboardPanel>> widgets = new ArrayList<>(tableBuilder.columnCount() * tableBuilder.rowCount());

        for (int rowIdx = 0; rowIdx < tableBuilder.rowCount(); ++rowIdx) {
            for (int colIdx = 0; colIdx < tableBuilder.columnCount(); ++colIdx) {
                var cell = tableBuilder.getCellOrEmpty(rowIdx, colIdx);
                var content = cell.getContentOrNull();
                if (cell.state() == TableWithSpansBuilder.CellState.EMPTY) {
                    DashboardPanel emptyPanel = new DashboardPanel(DashboardPanel.Type.MARKDOWN, "", "", "", "",
                            cell.getRowspan(), cell.getColspan());
                    var emptyWidget = new WidgetScheme<>(colIdx, rowIdx, cell.getColspan(), cell.getRowspan(), emptyPanel);
                    widgets.add(emptyWidget);

                } else if (cell.state() == TableWithSpansBuilder.CellState.FULL) {
                    if (content == null) {
                        // Unreachable
                        continue;
                    }
                    var widget = new WidgetScheme<>(colIdx, rowIdx, cell.getColspan(), cell.getRowspan(), content);
                    widgets.add(widget);
                }
            }
        }
        return widgets;
    }

    public DashScheme<T> replace(WidgetScheme<T> oldWidget, DashScheme<T> insertedDashScheme) {
        int oldWidgetIndex = widgets.indexOf(oldWidget);
        if (oldWidgetIndex < 0) {
            return this;
        }

        int insertedDashHeight = insertedDashScheme.getHeight();
        int insertedDashWidth = insertedDashScheme.getWidth();
        int oldWidgetHeight = oldWidget.h;
        int oldWidgetWidth = oldWidget.w;

        if (insertedDashHeight == 0 || insertedDashWidth == 0) {
            List<WidgetScheme<T>> result = new ArrayList<>(widgets);
            result.remove(oldWidgetIndex);
            return new DashScheme<>(result);
        }

        DashScheme<T> oldTransformedWidgets = this.scale(insertedDashWidth, insertedDashHeight);

        int xShift = insertedDashWidth * oldWidget.x;
        int yShift = insertedDashHeight * oldWidget.y;
        DashScheme<T> newTransformedWidgets = insertedDashScheme.affine(xShift, yShift, oldWidgetWidth, oldWidgetHeight);

        List<WidgetScheme<T>> result = new ArrayList<>(oldTransformedWidgets.widgets);
        result.remove(oldWidgetIndex);
        result.addAll(oldWidgetIndex, newTransformedWidgets.widgets);

        return new DashScheme<>(result);
    }

    private int getHeight() {
        return widgets.stream().mapToInt(widget -> widget.y + widget.h).max().orElse(0);
    }

    private int getWidth() {
        return widgets.stream().mapToInt(widget -> widget.x + widget.w).max().orElse(0);
    }

    public DashScheme<T> reduceSizes() {
        if (widgets.isEmpty()) {
            return this;
        }

        int xCoordGcd = widgets.get(0).w;
        int yCoordGcd = widgets.get(0).h;
        for (int i = 1; i < widgets.size(); i++) {
            WidgetScheme<T> widget = widgets.get(i);
            xCoordGcd = ArithmeticUtils.gcd(xCoordGcd, widget.x);
            xCoordGcd = ArithmeticUtils.gcd(xCoordGcd, widget.x + widget.w);
            yCoordGcd = ArithmeticUtils.gcd(yCoordGcd, widget.y);
            yCoordGcd = ArithmeticUtils.gcd(yCoordGcd, widget.y + widget.h);
        }

        if (xCoordGcd == 1 && yCoordGcd == 1) {
            return this;
        }

        int yCoef = yCoordGcd;
        int xCoef = xCoordGcd;

        List<WidgetScheme<T>> result = widgets.stream()
                .map(widget -> new WidgetScheme<>(
                        widget.x / xCoef,
                        widget.y / yCoef,
                        widget.w / xCoef,
                        widget.h / yCoef,
                        widget.content)
                )
                .collect(Collectors.toList());

        return new DashScheme<>(result);
    }

    public DashScheme<T> scaleByHeight(int widgetHeight) {
        List<WidgetScheme<T>> result = widgets.stream()
                .map(widget -> new WidgetScheme<>(
                        widget.x,
                        widget.y * widgetHeight,
                        widget.w,
                        widget.h * widgetHeight,
                        widget.content)
                )
                .collect(Collectors.toList());

        return new DashScheme<>(result);
    }

    public DashScheme<T> scaleByFixedWidth(int fixedWidth) {
        int dashWidth = getWidth();

        if (dashWidth == 0) {
            return this;
        }

        if (dashWidth < fixedWidth) {
            if (dashWidth < fixedWidth / 2) {
                int xCoef = fixedWidth / dashWidth;
                var dashScheme = scale(xCoef, 1);
                if (dashScheme.getWidth() == fixedWidth) {
                    return dashScheme;
                } else {
                    return dashScheme.scaleDisproportionately(fixedWidth);
                }
            } else {
                return this.scaleDisproportionately(fixedWidth);
            }
        }

        // More than <fixedWidth> width, we can't scale it.
        return this;
    }

    private DashScheme<T> scaleDisproportionately(int maxWidth) {
        int dashWidth = this.getWidth();
        int diffCoef = maxWidth - dashWidth;

        var newWidgets = widgets.stream().map(widget -> {
            final int newX;
            final int newW;

            if (widget.x + widget.w <= diffCoef) {
                newX = widget.x * 2;
                newW = widget.w * 2;
            } else if (widget.x > diffCoef) {
                newX = widget.x + diffCoef;
                newW = widget.w;
            } else {
                int width1 = diffCoef - widget.x;
                int width2 = widget.w - width1;
                newX = widget.x * 2;
                newW = 2 * width1 + width2;
            }

            return new WidgetScheme<>(
                    newX,
                    widget.y,
                    newW,
                    widget.h,
                    widget.content
            );
        })
        .collect(Collectors.toList());

        return new DashScheme<>(newWidgets);
    }

    private DashScheme<T> scale(int xCoef, int yCoef) {
        List<WidgetScheme<T>> result = widgets.stream()
                .map(widget -> new WidgetScheme<>(
                                widget.x * xCoef,
                                widget.y * yCoef,
                                widget.w * xCoef,
                                widget.h * yCoef,
                                widget.content
                        )
                ).collect(Collectors.toList());

        return new DashScheme<>(result);
    }

    private DashScheme<T> affine(int xShift, int yShift, int xCoef, int yCoef) {
        List<WidgetScheme<T>> result = widgets.stream()
                .map(widget -> new WidgetScheme<>(
                                xShift + widget.x * xCoef,
                                yShift + widget.y * yCoef,
                                widget.w * xCoef,
                                widget.h * yCoef,
                                widget.content
                        )
                ).collect(Collectors.toList());

        return new DashScheme<>(result);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DashScheme<?> that = (DashScheme<?>) o;
        return widgets.equals(that.widgets);
    }

    @Override
    public int hashCode() {
        return Objects.hash(widgets);
    }

    @Override
    public String toString() {
        return "DashScheme{" +
                "widgets=" + widgets +
                '}';
    }

    public static class WidgetScheme<T> {
        public final int x;
        public final int y;
        public final int w;
        public final int h;
        public final T content;

        public WidgetScheme(int x, int y, int w, int h, T content) {
            this.x = x;
            this.y = y;
            this.w = w;
            this.h = h;
            this.content = content;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            WidgetScheme<?> that = (WidgetScheme<?>) o;
            return x == that.x &&
                    y == that.y &&
                    w == that.w &&
                    h == that.h &&
                    content.equals(that.content);
        }

        @Override
        public int hashCode() {
            return Objects.hash(x, y, w, h, content);
        }

        @Override
        public String toString() {
            return "WidgetScheme{" +
                    "x=" + x +
                    ", y=" + y +
                    ", w=" + w +
                    ", h=" + h +
                    ", content=" + content +
                    '}';
        }
    }
}
