package ru.yandex.solomon.gateway.entityConverter;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.v3.IframeWidget;
import ru.yandex.monitoring.v3.TextWidget;
import ru.yandex.monitoring.v3.TitleWidget;
import ru.yandex.monitoring.v3.Widget;
import ru.yandex.solomon.config.protobuf.frontend.EntityConverterConfig;
import ru.yandex.solomon.core.db.model.DashboardPanel;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.url.UrlUtils;
import ru.yandex.solomon.labels.LabelKeys;

public class DashWidgetConverter {
    private static final Pattern JS_TITLE_PATTERN = Pattern.compile("javascript:'<h1.*?>(.*?)</h1>'");

    public static CompletableFuture<ParsedWidgetContent> convertWidgetAsync(
            String widgetId,
            DashboardPanel panel,
            Map<String, String> overrideParams,
            ExternalLoader externalLoader,
            EntityConverterConfig config)
    {
        if (panel.getType() == DashboardPanel.Type.MARKDOWN) {
            return CompletableFuture.completedFuture(setWidgetFromOldMarkdownPanel(panel, config));
        }

        return convertToWidgetFromOldIframePanelAsync(widgetId, panel, overrideParams, externalLoader, config);
    }

    private static CompletableFuture<ParsedWidgetContent> convertToWidgetFromOldIframePanelAsync(
            String widgetId,
            DashboardPanel panel,
            Map<String, String> overrideParams,
            ExternalLoader externalLoader,
            EntityConverterConfig config)
    {
        String widgetTitle = makeWidgetTitle(panel, overrideParams);

        String rawUrl = panel.getUrl();
        String url = fixUrl(rawUrl, overrideParams);

        if (url.isEmpty()) {
            // Empty panel.
            return CompletableFuture.completedFuture(new ParsedWidgetContent.SimpleWidget(Widget.newBuilder()
                    .setText(TextWidget.getDefaultInstance())
                    .build()));
        }

        if (url.startsWith("trigger-states")) {
            // Legacy from old triggers (about 49 panels).
            return CompletableFuture.completedFuture(null);
        }

        if (url.startsWith("<iframe")) {
            // Not supported (https://solomon.yandex-team.ru/?project=talents&dashboard=talents_full_dashboard).
            return CompletableFuture.completedFuture(null);
        }

        if (url.startsWith("javascript:")) {
            // Most of js panels are <h1 style="...">...</h1>, we may convert to title widgets.
            return CompletableFuture.completedFuture(tryToConvertJsPanel(url));
        }

        if (url.startsWith("/admin") || url.startsWith("admin")) {
            if (url.contains("/singlestat")) {
                // Add singlestat widget.
                return AlertWidgetConverter.convertToAlertSinglestatWidgetAsync(widgetTitle, url, externalLoader);
            } else if (url.contains("/subAlerts")) {
                // Add sub alerts widget.
                return CompletableFuture.completedFuture(AlertWidgetConverter.convertToSubAlertsWidget(widgetTitle, url));
            } else if (url.contains("/alertStates")) {
                // Add alert states complex widget.
                return CompletableFuture.completedFuture(AlertWidgetConverter.convertToAlertStatesWidget(widgetTitle, url));
            }

            // Unknown admin link (zero cases now).
            return CompletableFuture.completedFuture(null);
        }

        if (url.startsWith("?")) {
            if (url.contains("&graph=auto&") || url.startsWith("?graph=auto") || url.endsWith("&graph=auto")) {
                // Auto graph widget.
                return CompletableFuture.completedFuture(AutoGraphWidgetConverter.convertToAutoGraphWidget(widgetTitle, url, widgetId));
            } else if (url.contains("&graph=") || (url.contains("?graph="))) {
                // Configured graph widget.
                return ConfiguredGraphWidgetConverter.convertToConfiguredGraphWidgetAsync(widgetTitle, url, widgetId, externalLoader);
            } else if (url.contains("&dashboard=") || url.contains("?dashboard=")) {
                // Dashboard panel complex widget.
                return convertToInlineDashboardWidgetAsync(url, externalLoader, config);
            }

            // Unknown graph link (about 17 empty panels without graph or dashboard).
            return CompletableFuture.completedFuture(null);
        }

        // Add iframe
        return CompletableFuture.completedFuture(setIframeWidget(widgetTitle, url));
    }

    private static ParsedWidgetContent setWidgetFromOldMarkdownPanel(DashboardPanel panel, EntityConverterConfig config) {
        var markdown = prepareMarkdown(panel.getMarkdown(), config);
        var widget = Widget.newBuilder().setText(TextWidget.newBuilder()
                .setText(markdown)
                .build())
                .build();

        return new ParsedWidgetContent.SimpleWidget(widget);
    }

    private static String prepareMarkdown(String markdown, EntityConverterConfig config) {
        String prefix = config.getTargetSiteUrl();
        return markdown.replaceAll("\\(/admin", "(" + prefix + "/admin")
                .replaceAll("\\(/?\\?project=", "(" + prefix + "/?project=")
                .replaceAll("\\(/?\\?cluster=", "(" + prefix + "/?cluster=");
    }

    public static String fixUrl(String rawUrl, Map<String, String> exactParams) {
        String url = rawUrl.trim();
        url = StringInterpolator.interpolatePattern(url, exactParams);
        url = StringUtils.removeStart(url, "https://solomon.yandex-team.ru");
        url = StringUtils.removeStart(url, "http://solomon.yandex-team.ru");
        url = StringUtils.removeStart(url, "solomon.yandex-team.ru");
        url = StringUtils.removeStart(url, "/#!");
        url = StringUtils.removeStart(url, "/#");
        url = StringUtils.removeStart(url, "//");
        url = StringUtils.removeStart(url, "/");
        url = StringUtils.removeStart(url, "/");
        return url;
    }

    private static ParsedWidgetContent setIframeWidget(String widgetTitle, String url) {
        var widget = Widget.newBuilder().setIframe(IframeWidget.newBuilder()
                .setTitle(widgetTitle)
                .setUrl(url)
                .build())
                .build();
        return new ParsedWidgetContent.SimpleWidget(widget);
    }

    @Nullable
    private static CompletableFuture<ParsedWidgetContent> convertToInlineDashboardWidgetAsync(
            String url,
            ExternalLoader externalLoader,
            EntityConverterConfig config)
    {
        var queryArgs = UrlUtils.parseUrlQueryArgs(url);
        var projectParam = queryArgs.getOrDefault(LabelKeys.PROJECT, "");
        var dashParam = queryArgs.getOrDefault(UserLinksBasic.DASHBOARD_PARAM, "");

        if (projectParam.isBlank()) {
            return CompletableFuture.completedFuture(null);
        }
        if (dashParam.isBlank()) {
            return CompletableFuture.completedFuture(null);
        }

        return externalLoader.loadDashboard(projectParam, dashParam)
                .thenCompose(oldDashboard -> {
                    if (oldDashboard == null) {
                        var widget = new ParsedWidgetContent.SimpleWidget(Widget.newBuilder()
                                .setText(TextWidget.newBuilder()
                                        .setText("Unknown dashboard: " + StringInterpolator.escapePattern(projectParam) + "/" + StringInterpolator.escapePattern(dashParam))
                                        .build())
                                .build());
                        return CompletableFuture.completedFuture(widget);
                    }

                    var extractedParameters = ExtractedParameters.from(oldDashboard.getProjectId(), oldDashboard.getParameters(), queryArgs);
                    return DashConverter.convertOldDashboardToDashSchemeAsync(oldDashboard, externalLoader, extractedParameters, "", config)
                            .thenApply(ParsedWidgetContent.ComplexWidget::new);
                });
    }

    @Nullable
    private static ParsedWidgetContent tryToConvertJsPanel(String url) {
        // Most of JS panels contain "javascript:'<h1>Title</h1>'" syntax,
        // so we try to convert to title widget.
        var matcher = JS_TITLE_PATTERN.matcher(url);
        if (matcher.matches()) {
            var title = matcher.group(1);
            var widget = Widget.newBuilder().setTitle(TitleWidget.newBuilder()
                    .setText(title)
                    .setSize(TitleWidget.TitleSize.TITLE_SIZE_L)
                    .build())
                    .build();
            return new ParsedWidgetContent.SimpleWidget(widget);
        }

        // In other cases we don't add JS panels.
        return null;
    }

    private static String makeWidgetTitle(DashboardPanel panel, Map<String, String> exactParams) {
        return StringInterpolator.interpolatePattern(panel.getTitle().trim() + " " + panel.getSubtitle().trim(), exactParams).trim();
    }
}
