package ru.yandex.solomon.gateway.entityConverter;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.mutable.MutableInt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monitoring.v3.QuickLinks;
import ru.yandex.solomon.config.protobuf.frontend.EntityConverterConfig;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.ProjectMenu;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class EntityConverter {
    private static final Logger logger = LoggerFactory.getLogger(EntityConverter.class);

    private volatile int DASHBOARD_CONVERSION_INFLIGHT = 1;
    private final EntityConverterConfig config;
    private final EntityUpdaterService entityUpdaterService;
    private final ExternalLoader externalLoader;
    private final DebugInfo debugInfo;
    private final Map<String, LastVersions> versionsMap;

    public EntityConverter(EntityConverterConfig config, ExternalLoader externalLoader, ObjectMapper objectMapper, EntityUpdaterService entityUpdaterService) {
        this.config = config;
        this.externalLoader = externalLoader;
        debugInfo = new DebugInfo(objectMapper);
        versionsMap = new HashMap<>();
        this.entityUpdaterService = entityUpdaterService;
    }

    public void runConverter() {
        if (config.getConvertAll()) {
            try {
                logger.debug("Read all project IDs");
                Set<String> allProjectIds = externalLoader.loadProjectIds();
                List<String> projectIds = allProjectIds.stream()
                        .filter(projectId -> !projectId.equals("yandexcloud_bootstrap") && !projectId.startsWith("yasm_"))
                        .sorted()
                        .collect(Collectors.toList());
                int totalCount = projectIds.size();
                int projectIdx = 0;

                for (String projectId : projectIds) {
                    projectIdx++;
                    logger.debug("Convert entities in project " + projectId + " (" + projectIdx + "/" + totalCount + ")");
                    ProjectEntityFilter filter = ProjectEntityFilter.all(projectId);
                    convertProjectEntities(filter);
                }
            } catch (Exception e) {
                logger.error("Conversion was failed", e);
                throw e;
            }
        } else {
            List<ProjectEntityFilter> filters = ConvertFilterParser.parse(config.getConvert());
            int totalCount = filters.size();
            int filterIdx = 0;

            for (var filter : filters) {
                filterIdx++;
                logger.debug("Convert " + filter.format() + " (" + filterIdx + "/" + totalCount + ")");
                convertProjectEntities(filter);
            }
        }
    }

    private void convertProjectEntities(ProjectEntityFilter filter) {
        String projectId = filter.projectId;
        final ParsedProjectSettings settings;
        if (config.getIgnoreProjectDisables()) {
            settings = new ParsedProjectSettings(false, false);
        } else {
            logger.debug("Load project " + projectId + " settings...");
            settings = externalLoader.loadProjectSettings(projectId).join();
        }

        if (filter.mode == ProjectEntityFilter.Mode.CONCRETE_DASHBOARD) {
            convertConcreteDashboard(projectId, filter.entityId, filter.widgetId, settings);
        } else if (filter.mode == ProjectEntityFilter.Mode.CONCRETE_GRAPH) {
            convertConcreteGraph(projectId, filter.entityId, settings);
        } else if (filter.mode == ProjectEntityFilter.Mode.MENU) {
            convertProjectMenu(projectId, settings);
        } else if (filter.mode == ProjectEntityFilter.Mode.GRAPHS_DASHBOARDS) {
            convertProjectGraphsAndDashboards(projectId, settings);
        } else if (filter.mode == ProjectEntityFilter.Mode.ALL) {
            convertAllProjectEntities(projectId, settings);
        } else {
            throw new IllegalStateException("unknown filter mode for project " + projectId + ": " + filter.mode);
        }
    }

    private void convertProjectGraphsAndDashboards(String projectId, ParsedProjectSettings projectSettings) {
        if (projectSettings.isDisableDashboardConvert()) {
            logger.debug("Automatic graph and dashboard convert is disabled in project " + projectId + ". Use --ignore-project-disables to prevent it.");
            return;
        }
        Collection<Graph> graphs = externalLoader.loadProjectGraphs(projectId);
        Collection<Dashboard> dashboards = externalLoader.loadProjectDashboards(projectId);
        convertOldGraphsAndDashboards(projectId, graphs, dashboards, null);
    }

    private void convertProjectMenu(String projectId, ParsedProjectSettings projectSettings) {
        try {
            if (projectSettings.isDisableMenuConvert()) {
                logger.debug("Automatic menu convert is disabled in project " + projectId + ". Use --ignore-project-disables to prevent it.");
                return;
            }
            var projectMenu = externalLoader.loadProjectMenu(projectId).join();
            if (projectMenu == null) {
                logger.debug("Failed to load menu in project " + projectId);
                return;
            }
            var quickLinks = QuickLinksConverter.convertAsync(projectMenu, config, externalLoader).join();
            debugInfo.saveDebugComparison(projectId, projectMenu, quickLinks);
            entityUpdaterService.saveQuickLinks(projectId, quickLinks);
            logger.debug("Project menu " + projectId + " converted");
        } catch (Exception e) {
            logger.error("Failed to convert quick links " + projectId, e);
        }
    }

    private void convertConcreteGraph(
            String projectId,
            String graphId,
            ParsedProjectSettings projectSettings)
    {
        try {
            if (projectSettings.isDisableDashboardConvert()) {
                logger.debug("Automatic graph convert is disabled in project " + projectId + ". Use --ignore-project-disables to prevent it.");
                return;
            }
            var graph = externalLoader.loadGraph(projectId, graphId).join();
            if (graph == null) {
                logger.debug("Failed to load graph " + projectId + "/" + graphId);
                return;
            }
            var newDashboard = GraphConverter.convertToDashboard(graph, config);
            debugInfo.saveDebugComparison(projectId, graph, newDashboard);
            entityUpdaterService.saveDashboard(newDashboard);
            logger.debug("Graph " + projectId + "/" + graphId + " converted");
        } catch (Exception e) {
            logger.error("Failed to convert graph " + projectId + "/" + graphId, e);
        }
    }

    private void convertConcreteDashboard(
            String projectId,
            String dashboardId,
            String widgetFilter,
            ParsedProjectSettings projectSettings)
    {
        try {
            if (projectSettings.isDisableDashboardConvert()) {
                logger.debug("Automatic dashboard convert is disabled in project " + projectId + ". Use --ignore-project-disables to prevent it.");
                return;
            }
            var dashboard = externalLoader.loadDashboard(projectId, dashboardId).join();
            if (dashboard == null) {
                logger.debug("Failed to load dashboard " + projectId + "/" + dashboardId);
                return;
            }
            var newDashboard = DashConverter.convertAsync(dashboard, externalLoader, widgetFilter, config).join();
            debugInfo.saveDebugComparison(projectId, dashboard, newDashboard);
            entityUpdaterService.saveDashboard(newDashboard);
            logger.debug("Dashboard " + projectId + "/" + dashboardId + (widgetFilter.isEmpty() ? "" : " with filter " + widgetFilter) + " converted");
        } catch (Exception e) {
            logger.error("Failed to convert dashboard " + projectId + "/" + dashboardId, e);
        }
    }

    private void convertAllProjectEntities(
            String projectId,
            ParsedProjectSettings projectSettings)
    {
        if (projectSettings.isDisableMenuConvert() && projectSettings.isDisableDashboardConvert()) {
            logger.debug("Automatic convert is disabled in project " + projectId + ". Use --ignore-project-disables to prevent it.");
            return;
        }
        ProjectMenu projectMenu = externalLoader.loadProjectMenu(projectId).join();
        if (projectMenu == null) {
            logger.debug("Failed to load menu in project " + projectId);
            return;
        }
        Collection<Graph> graphs = externalLoader.loadProjectGraphs(projectId);
        Collection<Dashboard> dashboards = externalLoader.loadProjectDashboards(projectId);

        LastVersions.Diff diff = null;
        if (config.getCheckVersion()) {
            LastVersions next = LastVersions.fromEntities(graphs, dashboards, projectMenu.getVersion());
            LastVersions prev = versionsMap.getOrDefault(projectId, LastVersions.create());
            if (!prev.isEmpty()) {
                diff = LastVersions.createDiff(prev, next);
            }
            versionsMap.put(projectId, next);
        }

        if (!projectSettings.isDisableDashboardConvert()) {
            convertOldGraphsAndDashboards(projectId, graphs, dashboards, diff);
        }

        if (!projectSettings.isDisableMenuConvert() && (diff == null || diff.changedProjectMenu)) {
            convertProjectMenu(projectId, projectMenu);
        }
    }

    private void convertOldGraphsAndDashboards(
            String projectId,
            Collection<Graph> graphs,
            Collection<Dashboard> dashboards,
            @Nullable LastVersions.Diff diff)
    {
        logger.debug("Create new dashboards...");

        if (!dashboards.isEmpty()) {
            Iterator<Dashboard> iterator = dashboards.iterator();
            MutableInt dashboardIndex = new MutableInt(0);
            AsyncActorBody body = () -> {
                while (true) {
                    if (!iterator.hasNext()) {
                        return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                    }
                    dashboardIndex.increment();
                    var dashboard = iterator.next();
                    String dashboardId = dashboard.getId();
                    if (diff != null && !diff.changedDashboardIds.contains(dashboardId)) {
                        return CompletableFuture.completedFuture(null);
                    }
                    logger.debug("Dashboard " + projectId + "/" + dashboardId + " (" + dashboardIndex.getValue() + "/" + dashboards.size() + ")");
                    try {
                       return DashConverter.convertAsync(dashboard, externalLoader, "", config)
                               .thenCompose(newDashboard -> {
                                   debugInfo.saveDebugComparison(projectId, dashboard, newDashboard);
                                   return entityUpdaterService.saveDashboard(newDashboard);
                               });
                    } catch (Exception e) {
                        logger.error("Failed to convert dashboard " + projectId + "/" + dashboardId, e);
                    }
                    return CompletableFuture.completedFuture(null);
                }
            };
            var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), DASHBOARD_CONVERSION_INFLIGHT);
            runner.start().join();
        }

        logger.debug("Create new graph dashboards...");

        if (!graphs.isEmpty()) {
            Iterator<Graph> iterator = graphs.iterator();
            MutableInt graphIndex = new MutableInt(0);
            AsyncActorBody body = () -> {
                while (true) {
                    if (!iterator.hasNext()) {
                        return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                    }
                    graphIndex.increment();
                    var graph = iterator.next();
                    String graphId = graph.getId();
                    if (diff != null && !diff.changedGraphIds.contains(graphId)) {
                        return CompletableFuture.completedFuture(null);
                    }
                    logger.debug("Graph " + projectId + "/" + graphId + " (" + graphIndex.getValue() + "/" + graphs.size() + ")");
                    try {
                        var newDashboard = GraphConverter.convertToDashboard(graph, config);
                        debugInfo.saveDebugComparison(projectId, graph, newDashboard);
                        return entityUpdaterService.saveDashboard(newDashboard);
                    } catch (Exception e) {
                        logger.error("Failed to convert graph " + projectId + "/" + graphId, e);
                    }
                    return CompletableFuture.completedFuture(null);
                }
            };
            var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), DASHBOARD_CONVERSION_INFLIGHT);
            runner.start().join();
        }
    }

    private void convertProjectMenu(
            String projectId,
            ProjectMenu projectMenu)
    {
        logger.debug("Set or update quick links...");
        try {
            QuickLinks newQuickLinks = QuickLinksConverter.convertAsync(projectMenu, config, externalLoader).join();
            debugInfo.saveDebugComparison(projectId, projectMenu, newQuickLinks);
            entityUpdaterService.saveQuickLinks(projectId, newQuickLinks);
        } catch (Exception e) {
            logger.error("Failed to convert project menu for project " + projectId, e);
        }
    }
}
