package ru.yandex.solomon.gateway.entityConverter;

import java.time.Clock;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.frontend.EntityConverterConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.ProjectMenuManager;
import ru.yandex.solomon.core.db.dao.DashboardsDao;
import ru.yandex.solomon.core.db.dao.GraphsDao;
import ru.yandex.solomon.core.db.dao.ProjectSettingsDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.gateway.api.v3alpha.MonitoringDashboardService;
import ru.yandex.solomon.gateway.api.v3alpha.MonitoringQuickLinksService;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.roles.LockContext;
import ru.yandex.solomon.spring.ConditionalOnBean;

/**
 * @author Alexey Trushkin
 */
@Import({
        LockContext.class,
})
@ConditionalOnBean(EntityConverterConfig.class)
@Configuration
public class EntityConverterContext {

    @Bean
    public EntityUpdaterService entityUpdaterService(
            MonitoringQuickLinksService monitoringQuickLinksService,
            MonitoringDashboardService monitoringDashboardService,
            MonitoringDashboardsDao dashboardsDao
    ) {
        return new EntityUpdaterService(dashboardsDao, monitoringQuickLinksService, monitoringDashboardService);
    }

    @Bean
    public ExternalLoader externalLoader(
            GraphsDao graphsDao,
            DashboardsDao dashboardsDao,
            ProjectMenuManager projectMenuManager,
            ProjectSettingsDao projectSettingsDao,
            AlertApi alertApi,
            ProjectsDao projectsDao
    ) {
        return new ExternalLoaderImpl(graphsDao, dashboardsDao, projectMenuManager, projectSettingsDao, alertApi, projectsDao);
    }

    @Bean
    @Nullable
    public EntityConverterScheduler entityConverterScheduler(
            EntityConverterConfig config,
            @Qualifier("gatewayLockService") LockService lockService,
            ThreadPoolProvider threads,
            MetricRegistry registry,
            ObjectMapper objectMapper,
            EntityUpdaterService entityUpdaterService,
            ExternalLoader externalLoader)
    {
        if (config == null) {
            return null;
        }

        var lock = lockService.distributedLock(EntityConverterScheduler.LOCK_NAME);
        var timer = threads.getSchedulerExecutorService();
        return new EntityConverterScheduler(
                config,
                lock,
                threads.getExecutorService(config.getThreadPoolName(), ""),
                timer,
                Clock.systemUTC(),
                registry,
                TimeUnitConverter.millis(config.getSchedulerInterval()),
                objectMapper,
                entityUpdaterService,
                externalLoader);
    }
}
