package ru.yandex.solomon.gateway.entityConverter;

import java.time.Clock;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.frontend.EntityConverterConfig;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Alexey Trushkin
 */
public class EntityConverterScheduler implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(EntityConverterScheduler.class);
    public static final String LOCK_NAME = "EntityConverterSchedulerMaster";

    private final EntityConverter entityConverter;
    private final DistributedLock lock;
    private final Clock clock;
    private final long initialDelayMillis;
    private final long minReindexPeriod;
    private volatile long latestReindexTs;
    private volatile boolean closed;
    private final ScheduledExecutorService timer;
    private final PingActorRunner actor;
    private final AsyncMetrics metrics;

    public EntityConverterScheduler(
            EntityConverterConfig config,
            DistributedLock lock,
            ExecutorService executor,
            ScheduledExecutorService timer,
            Clock clock,
            MetricRegistry registry,
            long intervalMillis,
            ObjectMapper objectMapper,
            EntityUpdaterService entityUpdaterService,
            ExternalLoader externalLoader)
    {
        this.entityConverter = new EntityConverter(config, externalLoader, objectMapper, entityUpdaterService);
        this.lock = lock;
        this.clock = clock;
        this.initialDelayMillis = Duration.ofMinutes(1).toMillis();
        minReindexPeriod = intervalMillis / 2;
        this.timer = timer;
        this.metrics = new AsyncMetrics(registry, "entity_converter.update");
        this.actor = PingActorRunner.newBuilder()
                .executor(executor)
                .timer(timer)
                .operation("Convert monitoring entities")
                .pingInterval(Duration.ofMillis(intervalMillis))
                .backoffDelay(Duration.ofMinutes(1))
                .onPing(this::act)
                .build();
        acquireLock();
    }

    public CompletableFuture<Void> act(int attempt) {
        if (closed) {
            return completedFuture(null);
        }

        if (!lock.isLockedByMe()) {
            return completedFuture(null);
        }
        long reindexTs = clock.millis();
        if (minReindexPeriod > reindexTs - latestReindexTs) {
            return completedFuture(null);
        }

        long startTimeNanos = System.nanoTime();
        metrics.callStarted(1);
        Throwable tempThrowable = null;
        try {
            entityConverter.runConverter();
        } catch (Throwable t) {
            tempThrowable = t;
        } finally {
            long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
            if (tempThrowable != null) {
                metrics.callCompletedError(elapsedMillis, 1);
            } else {
                metrics.callCompletedOk(elapsedMillis, 1);
            }
            latestReindexTs = reindexTs;
        }
        return tempThrowable == null ? completedFuture(null) : CompletableFuture.failedFuture(tempThrowable);
    }

    public void schedule() {
        long delay = DurationUtils.randomize(initialDelayMillis);
        timer.schedule(actor::forcePing, delay, TimeUnit.MILLISECONDS);
    }

    private void acquireLock() {
        if (closed) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Acquire {} lock, seqNo {}", LOCK_NAME, seqNo);
                schedule();
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Loose {} lock by reason: {}", LOCK_NAME, reason);
                acquireLock();
            }
        }, 5, TimeUnit.MINUTES);
    }

    @Override
    public void close() {
        closed = true;
        actor.close();
    }
}
