package ru.yandex.solomon.gateway.entityConverter;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monitoring.v3.Dashboard;
import ru.yandex.monitoring.v3.GetDashboardRequest;
import ru.yandex.monitoring.v3.GetQuickLinksRequest;
import ru.yandex.monitoring.v3.QuickLinks;
import ru.yandex.monitoring.v3.UpdateQuickLinksRequest;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.conf.db3.MonitoringDashboardsDao;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.api.v3alpha.MonitoringDashboardService;
import ru.yandex.solomon.gateway.api.v3alpha.MonitoringQuickLinksService;
import ru.yandex.solomon.gateway.api.v3alpha.dao.ydb.MonitoringDashboardConverter;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
class EntityUpdaterService {
    private static final Logger logger = LoggerFactory.getLogger(EntityUpdaterService.class);

    private final MonitoringDashboardsDao dashboardsDao;
    private final MonitoringQuickLinksService quickLinksService;
    private final MonitoringDashboardService monitoringDashboardService;

    public EntityUpdaterService(
            MonitoringDashboardsDao dashboardsDao,
            MonitoringQuickLinksService quickLinksService,
            MonitoringDashboardService monitoringDashboardService)
    {
        this.dashboardsDao = dashboardsDao;
        this.monitoringDashboardService = monitoringDashboardService;
        this.quickLinksService = quickLinksService;
    }

    public CompletableFuture<Void> saveDashboard(Dashboard newDashboard) {
        String generatedId = newDashboard.getId();
        if (StringUtils.isEmpty(generatedId)) {
            generatedId = IdGenerator.generateInternalId();
            logger.error("Failed to find generated id, create new: " + generatedId + " (" + newDashboard.getDescription() + ")");
            newDashboard = newDashboard.toBuilder()
                    .setId(generatedId)
                    .setVersion(0)
                    .build();
        }
        return saveDashboardInner(newDashboard);
    }

    private CompletableFuture<Void> saveDashboardInner(Dashboard newDashboard) {
        return monitoringDashboardService.getDashboard(GetDashboardRequest.newBuilder()
                .setDashboardId(newDashboard.getId())
                .build())
                .thenCompose(dashboard -> {
                    if (dashboard.isEmpty()) {
                        return insertDashboard(newDashboard);
                    } else {
                        return insertDashboard(newDashboard.toBuilder()
                                .setVersion(dashboard.get().getVersion())
                                .build());
                    }
                })
                .whenComplete((quickLinks, throwable) -> {
                    if (throwable != null) {
                        logger.error("failed to save dashboard " + newDashboard.getProjectId() + "/" + newDashboard.getId() + " in storage", throwable);
                    }
                });
    }

    private CompletableFuture<Void> insertDashboard(Dashboard newDashboard) {
        var request = ru.yandex.monitoring.v3.priv.AdminUpsertDashboardRequest.newBuilder()
                .setDashboard(newDashboard)
                .build();
        return dashboardsDao.upsert(MonitoringDashboardConverter.encode(request.getDashboard(), ""));
    }

    public void saveQuickLinks(String projectId, QuickLinks newQuickLinks) {
        quickLinksService.getQuickLinks(GetQuickLinksRequest.newBuilder()
                .setProjectId(projectId)
                .build())
                .thenCompose(quickLinks -> quickLinksService.updateQuickLinks(UpdateQuickLinksRequest.newBuilder()
                        .setProjectId(newQuickLinks.getProjectId())
                        .setData(newQuickLinks.getData())
                        .setVersion(quickLinks.getVersion())
                        .build(), Account.ANONYMOUS))
                .whenComplete((quickLinks, throwable) -> {
                    if (throwable != null) {
                        logger.error("failed to save quick links " + projectId + " in storage", throwable);
                    }
                });
    }
}
