package ru.yandex.solomon.gateway.entityConverter;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.core.conf.ProjectMenuManager;
import ru.yandex.solomon.core.db.dao.DashboardsDao;
import ru.yandex.solomon.core.db.dao.GraphsDao;
import ru.yandex.solomon.core.db.dao.ProjectSettingsDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ProjectMenu;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.model.protobuf.Label;
import ru.yandex.solomon.ydb.page.PageOptions;
import ru.yandex.solomon.ydb.page.PagedResult;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ExternalLoaderImpl implements ExternalLoader {
    private final GraphsDao graphsDao;
    private final DashboardsDao dashboardsDao;
    private final ProjectMenuManager projectMenuManager;
    private final ProjectSettingsDao projectSettingsDao;
    private final ProjectsDao projectsDao;
    private final AlertApi alertApi;

    ExternalLoaderImpl(
            GraphsDao graphsDao,
            DashboardsDao dashboardsDao,
            ProjectMenuManager projectMenuManager,
            ProjectSettingsDao projectSettingsDao,
            AlertApi alertApi,
            ProjectsDao projectsDao)
    {
        this.graphsDao = graphsDao;
        this.dashboardsDao = dashboardsDao;
        this.projectMenuManager = projectMenuManager;
        this.projectSettingsDao = projectSettingsDao;
        this.alertApi = alertApi;
        this.projectsDao = projectsDao;
    }

    @Override
    public CompletableFuture<Graph> loadGraph(String projectId, String graphId) {
        return graphsDao.findOne(projectId, "", graphId)
                .handle((dto, throwable) -> {
                    if (throwable != null || dto.isEmpty()) {
                        return null;
                    }
                    return dto.get();
                });
    }

    @Override
    public CompletableFuture<Dashboard> loadDashboard(String projectId, String dashId) {
        return dashboardsDao.findOne(projectId, "", dashId)
                .handle((dto, throwable) -> {
                    if (throwable != null || dto.isEmpty()) {
                        return null;
                    }
                    return dto.get();
                });
    }

    @Override
    public CompletableFuture<ProjectMenu> loadProjectMenu(String projectId) {
        return projectMenuManager.getProjectMenu(projectId)
                .handle((dto, throwable) -> {
                    if (throwable != null) {
                        return null;
                    }
                    return dto;
                });
    }

    @Override
    public CompletableFuture<ParsedProjectSettings> loadProjectSettings(String projectId) {
        return projectSettingsDao.find(projectId)
                .handle((dto, throwable) -> {
                    if (throwable != null) {
                        return null;
                    }
                    boolean disableDashboardConvert = "true" .equals(dto.getSettings().get("disableDashboardConvert"));
                    boolean disableMenuConvert = "true" .equals(dto.getSettings().get("disableMenuConvert"));
                    return new ParsedProjectSettings(disableDashboardConvert, disableMenuConvert);
                });
    }

    @Override
    public CompletableFuture<Map<String, String>> loadSubAlertLabels(String projectId, String alertId, String subAlertId) {
        TReadSubAlertRequest request = TReadSubAlertRequest.newBuilder()
                .setProjectId(projectId)
                .setParentId(alertId)
                .setAlertId(subAlertId)
                .build();

        return alertApi.readSubAlert(request)
                .handle((dto, throwable) -> {
                    if (throwable != null) {
                        return null;
                    }
                    return dto.getSubAlert().getGroupKeyList()
                            .stream()
                            .collect(Collectors.toMap(
                                    Label::getKey,
                                    Label::getValue,
                                    (u, v) -> {
                                        throw new IllegalStateException("Duplicate key " + u);
                                    },
                                    () -> new LinkedHashMap<>(dto.getSubAlert().getGroupKeyCount())));
                });
    }

    @Override
    public Collection<Graph> loadProjectGraphs(String projectId) {
        return loadEntities(
                (page) -> graphsDao.findByProjectId(projectId, "", new PageOptions(1000, page), "").join(),
                (graph) -> graphsDao.findOne(projectId, "", graph.getId()).join(),
                Graph::getId);
    }

    @Override
    public Collection<Dashboard> loadProjectDashboards(String projectId) {
        return loadEntities(
                (page) -> dashboardsDao.findByProjectId(projectId, "", new PageOptions(1000, page), "").join(),
                (dashboard) -> dashboardsDao.findOne(projectId, "", dashboard.getId()).join(),
                Dashboard::getId);
    }

    @Override
    public Set<String> loadProjectIds() {
        return projectsDao.findAllNames().join().stream()
                .map(Project::getId)
                .collect(Collectors.toSet());
    }

    private <R> Collection<R> loadEntities(Function<Integer, PagedResult<R>> loader, Function<R, Optional<R>> entityloader, Function<R, String> idMapper) {
        Map<String, R> result = new HashMap<>(1000);
        loadEntitiesImpl(result, loader, entityloader, idMapper, 0);
        return result.values();
    }

    private <T, R> void loadEntitiesImpl(Map<String, R> result, Function<Integer, PagedResult<R>> loader, Function<R, Optional<R>> entityloader, Function<R, String> idMapper, int page) {
        PagedResult<R> apply = loader.apply(page);
        apply.getResult().iterator().forEachRemaining(el -> {
            Optional<R> entityOptional = entityloader.apply(el);
            entityOptional.ifPresent(r -> result.put(idMapper.apply(r), r));
        });

        if (apply.getResult().isEmpty()) {
            return;
        }

        loadEntitiesImpl(result, loader, entityloader, idMapper, page + 1);
    }
}
