package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monitoring.v3.LabelValuesParameter;
import ru.yandex.monitoring.v3.Parameter;
import ru.yandex.monitoring.v3.Parametrization;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class ExtractedParameters {
    public final String dataProjectId;
    public final Parametrization parametrization;
    public final Map<String, String> overrideParameters;

    private ExtractedParameters(String dataProjectId, Parametrization parametrization, Map<String, String> overrideParameters) {
        this.dataProjectId = dataProjectId;
        this.parametrization = parametrization;
        this.overrideParameters = overrideParameters;
    }

    public static void switchParamToFirstPlace(String paramName, List<Parameter> parameters) {
        for (int i = 0; i < parameters.size(); ++i) {
            Parameter parameter = parameters.get(i);
            if (parameter.getName().equals(paramName)) {
                parameters.remove(i);
                parameters.add(0, parameter);
                return;
            }
        }
    }

    public static ExtractedParameters from(
        String projectId,
        ru.yandex.solomon.core.db.model.Selector[] oldParameters,
        Map<String, String> queryArgs)
    {
        List<Selector> globSelectors = new ArrayList<>();
        Map<String, String> globParameters = new HashMap<>();

        String dataProjectId = projectId;

        for (var oldParameter : oldParameters) {
            String name = oldParameter.getName();
            String value = oldParameter.getValue();

            if (LabelKeys.PROJECT.equals(name)) {
                dataProjectId = value;
            } else if (value.startsWith("!")) {
                globParameters.put(name, value);
                globSelectors.add(Selector.notGlob(name, value.substring(1)));
            } else {
                globParameters.put(name, value);
                globSelectors.add(Selector.glob(name, value));
            }
        }

        final List<Parameter> newParameters;
        final String predefinedSelectorsStr;

        if (globSelectors.isEmpty()) {
            predefinedSelectorsStr = "";
            newParameters = List.of();
        } else {
            var predefinedSelectors = Selectors.builder();
            predefinedSelectors.add(Selector.exact(LabelKeys.PROJECT, dataProjectId));
            predefinedSelectors.addAll(globSelectors);
            predefinedSelectorsStr = Selectors.format(predefinedSelectors.build());

            newParameters = globSelectors.stream()
                    .map(globSelector -> Parameter.newBuilder()
                            .setName(globSelector.getKey())
                            .setTitle(globSelector.getKey())
                            .setHidden(false)
                            .setLabelValues(LabelValuesParameter.newBuilder()
                                    .setLabelKey(globSelector.getKey())
                                    .build())
                            .build())
                    .collect(Collectors.toList());
        }

        switchParamToFirstPlace("service", newParameters);
        switchParamToFirstPlace("cluster", newParameters);

        var parametrization = Parametrization.newBuilder()
                .addAllParameters(newParameters)
                .setSelectors(predefinedSelectorsStr)
                .build();

        // Make override selector params.
        Map<String, String> overrideParameters = new LinkedHashMap<>();
        overrideParameters.put(LabelKeys.PROJECT, dataProjectId);

        if (!queryArgs.isEmpty()) {
            for (var globParam : globParameters.entrySet()) {
                var name = globParam.getKey();
                var paramValue = globParam.getValue();
                var queryArgValue = queryArgs.getOrDefault(name, "");
                if (queryArgValue.isEmpty()) {
                    queryArgValue = queryArgs.getOrDefault(UserLinksBasic.LABEL_NAME_QA_PREFIX + name, "");
                }
                if (!queryArgValue.isEmpty()) {
                    overrideParameters.put(name, queryArgValue);
                } else {
                    overrideParameters.put(name, paramValue);
                }
            }
        } else {
            for (var globParamName : globParameters.keySet()) {
                overrideParameters.put(globParamName, "{{" + globParamName + "}}");
            }
        }

        return new ExtractedParameters(dataProjectId, parametrization, overrideParameters);
    }
}
