package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import ru.yandex.monitoring.v3.ChartWidget;
import ru.yandex.monitoring.v3.ChartWidget.Queries;
import ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides;
import ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.SeriesOverrideSettings;
import ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.SeriesVisualizationType;
import ru.yandex.monitoring.v3.Dashboard;
import ru.yandex.monitoring.v3.Parametrization;
import ru.yandex.monitoring.v3.Widget;
import ru.yandex.solomon.config.protobuf.frontend.EntityConverterConfig;
import ru.yandex.solomon.core.db.model.Selector;
import ru.yandex.solomon.core.db.model.graph.ElementTransform;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.core.db.model.graph.GraphElement;
import ru.yandex.solomon.core.db.model.graph.GraphElementType;
import ru.yandex.solomon.core.db.model.graph.YaxisPosition;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.conf.GraphSettings;
import ru.yandex.solomon.gateway.utils.conf.source.GraphSettingsSource;
import ru.yandex.solomon.gateway.utils.conf.source.GraphSettingsSourceMap;
import ru.yandex.solomon.gateway.utils.conf.source.GraphSettingsSourceSeq;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsBuilder;
import ru.yandex.solomon.util.Quoter;

import static ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.SeriesVisualizationType.SERIES_VISUALIZATION_TYPE_AREA;
import static ru.yandex.monitoring.v3.ChartWidget.SeriesOverrides.SeriesVisualizationType.SERIES_VISUALIZATION_TYPE_LINE;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class GraphConverter {
    // Copy from SelLexer#DOUBLE_PATTERN
    private static final Pattern DOUBLE_PATTERN = Pattern.compile("(?i)[-+]?[0-9]*\\.?[0-9]+([eE][-+]?[0-9]+)?[KMGTPE]?");

    public static ru.yandex.monitoring.v3.Dashboard convertToDashboard(Graph graph, EntityConverterConfig config) {
        var graphData = convertGraphData(graph, Map.of());

        String oldGraphUrl = config.getTargetSiteUrl() + "/admin/projects/" + graph.getProjectId() + "/graphs/" + graph.getId();
        String description = ("[GENERATED from " + oldGraphUrl + "] " + graph.getDescription()).trim();

        var inlineGraphWidget = graphData.inlineGraphWidgetBuilder
                .setId("00")
                .setTitle(graph.getName())
                .setDescription(description)
                .build();

        return Dashboard.newBuilder()
                .setId(graph.getGeneratedId())
                .setProjectId(graph.getProjectId())
                .setName("Configured graph " + StringInterpolator.interpolateName(graph.getName()))
                .setDescription(description)
                .setParametrization(graphData.parametrization)
                        .addWidgets(Widget.newBuilder()
                                .setPosition(Widget.LayoutPosition.newBuilder()
                                        .setH(DashConverter.MIN_HEIGHT)
                                        .setW(DashConverter.MAX_WIDTH)
                                        .build())
                                .setChart(inlineGraphWidget)
                                .build())
                .setCreatedAt(Timestamps.fromMillis(graph.getCreatedAtMillis()))
                .setCreatedBy(graph.getCreatedBy())
                .setUpdatedAt(Timestamps.fromMillis(graph.getUpdatedAtMillis()))
                .setUpdatedBy(graph.getUpdatedBy())
                .build();
    }

    public static ConvertedGraphData convertGraphData(Graph graph, Map<String, String> queryArgs) {
        // Extract info from parameters: data project ID, exact parameters and list of label values parameters.
        var extractedParameters = ExtractedParameters.from(graph.getProjectId(), graph.getParameters(), queryArgs);
        var dataProjectId = extractedParameters.dataProjectId;
        var parametrization = extractedParameters.parametrization;
        var overrideParams = extractedParameters.overrideParameters;

        // Make visualization settings.
        GraphSettingsSourceSeq source = mergeGraphAndQueryArgs(graph, queryArgs);

        String checks = queryArgs.getOrDefault(UserLinksBasic.CHECKS_QA, "");

        // Create targets and extract series overrides from elements.
        var triple = makeTargetsAndSeriesOverrides(graph, source, overrideParams, checks);
        var targets = triple.getLeft();
        var seriesOverrides = triple.getMiddle();
        var needToResetChecks = triple.getRight();

        // Try to not use series overrides if all series overrides has same stack params
        Boolean overrideStack = null;

        if (!seriesOverrides.isEmpty() && targets.size() == seriesOverrides.size()) {
            if (canConvertOverridesToStacks(seriesOverrides)) {
                seriesOverrides = cleanSeriesOverridesFromType(seriesOverrides);
                overrideStack = true;
            } else if (canConvertOverridesToLines(seriesOverrides)) {
                seriesOverrides = cleanSeriesOverridesFromType(seriesOverrides);
                overrideStack = false;
            }
        }

        ChartWidget.VisualizationSettings visualizationSettings = GraphUtils.makeVisualizationSettings(dataProjectId, source, overrideStack);

        var nameHidingSettings = needToResetChecks ?
                ChartWidget.NameHidingSettings.getDefaultInstance()
                : GraphUtils.makeNameHidingSettings(checks);

        // Make queries from targets and downsampling.
        var queriesBuilder = ChartWidget.Queries.newBuilder()
                .addAllTargets(targets);

        var downsampling = GraphUtils.makeDownsampling(source);
        queriesBuilder.setDownsampling(downsampling);
        var queries = queriesBuilder.build();

        var inlineGraphWidget = ChartWidget.newBuilder()
                .setQueries(queries)
                .setVisualizationSettings(visualizationSettings)
                .addAllSeriesOverrides(seriesOverrides)
                .setNameHidingSettings(nameHidingSettings);

        return new ConvertedGraphData(inlineGraphWidget, parametrization);
    }

    private static GraphSettingsSourceSeq mergeGraphAndQueryArgs(Graph graph, Map<String, String> queryArgs) {
        var graphSourceMap = GraphToSettingsConverter.convert(graph);
        var graphSource = new GraphSettingsSourceMap(graphSourceMap);
        var querySource = new GraphSettingsSourceMap(queryArgs);
        return new GraphSettingsSourceSeq(graphSource, querySource);
    }

    private static boolean canConvertOverridesToStacks(List<SeriesOverrides> seriesOverrides) {
        return allTargetsHasType(seriesOverrides, SERIES_VISUALIZATION_TYPE_AREA);
    }

    private static boolean canConvertOverridesToLines(List<SeriesOverrides> seriesOverrides) {
        return allTargetsHasType(seriesOverrides, SERIES_VISUALIZATION_TYPE_LINE);
    }

    private static List<SeriesOverrides> cleanSeriesOverridesFromType(List<SeriesOverrides> seriesOverrides) {
        return seriesOverrides.stream()
                .map(override -> {
                    SeriesOverrideSettings oldSettings = override.getSettings();
                    var newSettings = override.getSettings().toBuilder()
                            .clearStack()
                            .clearStackName()
                            .clearGrowDown()
                            .clearType()
                            .build();
                    return override.toBuilder()
                            .setSettings(newSettings)
                            .build();
                })
                .filter(override -> {
                    SeriesOverrideSettings settings = override.getSettings();
                    boolean isEmpty = settings.getColor().isEmpty()
                            && settings.getYaxisPosition() != SeriesOverrides.YaxisPosition.YAXIS_POSITION_RIGHT;
                    return !isEmpty;
                })
                .collect(Collectors.toList());
    }

    private static boolean allTargetsHasType(List<SeriesOverrides> seriesOverrides, SeriesVisualizationType type) {
        if (seriesOverrides.isEmpty()) {
            return false;
        }
        String stackName = null;
        for (var override : seriesOverrides) {
            if (override.getTypeCase() == SeriesOverrides.TypeCase.TARGET_INDEX) {
                SeriesOverrideSettings settings = override.getSettings();
                boolean isNeccessaryTarget =
                        settings.getType().equals(type)
                                && !settings.getGrowDown();
                if (!isNeccessaryTarget) {
                    return false;
                }
                if (stackName == null) {
                    stackName = settings.getStackName();
                } else if (!settings.getStackName().equals(stackName)) {
                    return false;
                }
            }
        }
        return true;
    }

    private static Triple<List<Queries.Target>, List<SeriesOverrides>, Boolean> makeTargetsAndSeriesOverrides(
            Graph graph,
            GraphSettingsSource source,
            Map<String, String> overrideParams,
            String checks)
    {
        List<ChartWidget.Queries.Target> elementTargets = new ArrayList<>();
        List<ChartWidget.SeriesOverrides> seriesOverrides = new ArrayList<>();

        var overLinesTransformParam = GraphSettings.overLinesTransform.getFromOrDefault(source).toLowerCase();
        var filter = GraphSettings.filter.getFromOrDefault(source).toLowerCase();
        String limit = GraphSettings.limit.getFromOrDefault(source);

        var hasOverLinesTransform = !overLinesTransformParam.equals("none");
        var hasFilter = !filter.equals("none");

        boolean hasOverLines = hasOverLinesTransform || hasFilter || !limit.isEmpty();

        // Initialize list of targets and series overrides.
        GraphElement[] elements = graph.getElements();
        for (int i = 0; i < elements.length; i++) {
            GraphElement element = elements[i];

            // Make target from element and some graph parameters.
            var target = elementToTarget(element, source, overrideParams);
            elementTargets.add(target);

            if (!hasOverLines) {
                // Make series overrides if element has specific area or color.
                SeriesOverrides seriesOverridesElement = makeSeriesOverrides(element, i);
                if (seriesOverridesElement != null) {
                    seriesOverrides.add(seriesOverridesElement);
                }
            }
        }

        // Make list of targets.
        var targetsAndNeedToResetChecks = makeTransformOverridesTarget(elementTargets, source, hasOverLines, checks);
        var targets = targetsAndNeedToResetChecks.getLeft();
        var needToResetChecks = targetsAndNeedToResetChecks.getRight();

        return Triple.of(targets, seriesOverrides, needToResetChecks);
    }

    @Nullable
    private static SeriesOverrides makeSeriesOverrides(GraphElement element, int elementIndex) {
        boolean hasSeriesOverrides = false;
        SeriesOverrideSettings.Builder overrideSettings = SeriesOverrideSettings.newBuilder();

        // Set color if needed.
        if (!element.getColor().isEmpty()) {
            overrideSettings.setColor(element.getColor());
            hasSeriesOverrides = true;
        }

        // Set right yaxis if needed.
        if (element.getYaxis() == YaxisPosition.RIGHT) {
            overrideSettings.setYaxisPosition(SeriesOverrides.YaxisPosition.YAXIS_POSITION_RIGHT);
            hasSeriesOverrides = true;
        }

        // Set area settings (area stack, stack name, grow type).
        if (element.getArea() != null) {
            if (element.getArea()) {
                overrideSettings.setType(SERIES_VISUALIZATION_TYPE_AREA);
            } else {
                overrideSettings.setType(SeriesVisualizationType.SERIES_VISUALIZATION_TYPE_LINE);
            }
            String stackName = element.getStack().isEmpty() ? "stack0" : element.getStack();
            overrideSettings.setStackName(stackName);
            if (element.getDown() != null) {
                overrideSettings.setGrowDown(element.getDown());
            }
            hasSeriesOverrides = true;
        } else {
            if (element.getStack().isEmpty()) {
                overrideSettings.setType(SERIES_VISUALIZATION_TYPE_LINE);
            } else {
                overrideSettings.setType(SERIES_VISUALIZATION_TYPE_AREA);
                overrideSettings.setStackName(element.getStack());
            }
            // Cratch to fix grow down graphs
            if (element.getDown() != null) {
                overrideSettings.setGrowDown(element.getDown());
            }
            hasSeriesOverrides = true;
        }

        SeriesOverrides seriesOverridesElement = null;
        if (hasSeriesOverrides) {
            seriesOverridesElement = SeriesOverrides.newBuilder()
                    .setSettings(overrideSettings)
                    .setTargetIndex(Integer.toString(elementIndex))
                    .build();
        }
        return seriesOverridesElement;
    }

    private static Pair<List<Queries.Target>, Boolean> makeTransformOverridesTarget(List<Queries.Target> targets, GraphSettingsSource source, boolean hasOverLines, String checks) {
        if (targets.isEmpty()) {
            return Pair.of(targets, false);
        }

        if (hasOverLines) {
            boolean needToResetChecks = false;

            String flattenExpression;
            if (targets.size() == 1) {
                // Simplify program for single element:
                // - {...} - for single selectors
                // - to_vector({...})  or to_vector(single({...})) - for expression
                var target = targets.get(0);
                var expr = target.getQuery();
                if (target.getTextMode()) {
                    flattenExpression = expr;
                } else {
                    flattenExpression = "to_vector(" + expr + ")";
                }
            } else {
                // Complex flatten expressions:
                // flatten(to_vector({...}), to_vector({...}), to_vector(single({...}))
                flattenExpression = "flatten(" + targets.stream().map(target -> "to_vector(" + target.getQuery() +
                        ")").collect(Collectors.joining(", ")) + ")";
            }

            String filter = GraphSettings.filter.getFromOrDefault(source).toLowerCase();
            String limit = GraphSettings.limit.getFromOrDefault(source);

            if (!limit.isEmpty()) {
                flattenExpression = "limit(" + flattenExpression + ", " + limit + ")";
            }

            if ("top".equals(filter) || "bottom".equals(filter)) {
                String filterLimit = GraphSettings.filterLimit.getFromOrDefault(source);
                String filterBy = GraphSettings.filterBy.getFromOrDefault(source).toLowerCase();
                flattenExpression = filter + "(" + filterLimit + ", \"" + filterBy + "\", " + flattenExpression + ")";
            }

            var overLinesTransform = GraphSettings.overLinesTransform.getFromOrDefault(source).toLowerCase();
            var percentilesStr = GraphSettings.percentiles.getFromOrDefault(source);
            var bucketLabel = GraphSettings.bucketLabel.getFromOrDefault(source);

            var percentiles = Arrays.stream(percentilesStr.split(","))
                    .map(String::trim)
                    .collect(Collectors.toList());

            final String rawExpr = flattenExpression;
            List<String> expressions;
            switch (overLinesTransform) {
                case "percentile": {
                    expressions = percentiles.stream()
                            .map(perc -> "percentile_group_lines(" + perc + ", " + rawExpr + ")")
                            .collect(Collectors.toList());
                    break;
                }
                case "weighted_percentile": {
                    expressions = percentiles.stream()
                            .map(perc -> "histogram_percentile(" + perc + ", " + (bucketLabel.isEmpty() ? "" :
                                    "\"" + bucketLabel + "\", ") + rawExpr + ")")
                            .collect(Collectors.toList());
                    break;
                }
                case "summary": {
                    var pair = QueryTransformations.makeSummary(rawExpr, checks);
                    expressions = pair.getLeft();
                    needToResetChecks = pair.getRight();
                    break;
                }
                default:
                    expressions = List.of(rawExpr);
            }

            boolean hideNoDataParam = GraphSettings.hideNoData.getBoolFrom(source);

            if (hideNoDataParam) {
                expressions = expressions.stream()
                        .map(expr -> "drop_empty_lines(" + expr + ")")
                        .collect(Collectors.toList());
            }

            var queries = expressions.stream()
                    .map(expr -> Queries.Target.newBuilder().setQuery(expr).setTextMode(true).build())
                    .collect(Collectors.toList());
            return Pair.of(queries, needToResetChecks);
        } else {
            boolean hideNoDataParam = GraphSettings.hideNoData.getBoolFrom(source);

            if (hideNoDataParam) {
                var queries = targets.stream()
                        .map(target -> target.toBuilder().setQuery("drop_empty_lines(" + target.getQuery() + ")").build())
                        .collect(Collectors.toList());
                return Pair.of(queries, false);
            }

            return Pair.of(targets, false);
        }
    }

    private static Queries.Target elementToTarget(
            GraphElement element,
            GraphSettingsSource source,
            Map<String, String> overrideParams) {
        boolean isSelectorsTarget = element.getType() == GraphElementType.SELECTORS;

        String expression;

        if (isSelectorsTarget) {
            Selectors parameterSelectors = oldParametersToNewSelectors(element.getSelectors());
            expression = Selectors.format(parameterSelectors);
        } else {
            expression = element.getExpression();
            if (DOUBLE_PATTERN.matcher(expression).matches()) {
                // Neccessary to support "40" or "40K" expressions for backward compatibility
                expression = "constant_line(" + expression + ")";
            }
        }

        expression = SelectorsReplacer.replaceSelectorsInCode(expression, overrideParams);

        var dropNans = GraphSettings.dropNans.getBoolFrom(source);
        if (dropNans) {
            expression = "drop_nan(" + expression + ")";
        }

        String transform;
        if (element.getTransform() != ElementTransform.NONE) {
            transform = element.getTransform().name().toLowerCase();
        } else {
            transform = GraphSettings.transform.getFromOrDefault(source).toLowerCase();
        }

        switch (transform) {
            case "none":
                break;
            case "differentiate": {
                expression = "non_negative_derivative(" + expression + ")";
                break;
            }
            case "differentiate_with_negative": {
                expression = "derivative(" + expression + ")";
                break;
            }
            case "integrate": {
                expression = "integrate_fn(" + expression + ")";
                break;
            }
            case "moving_average": {
                var movingWindowParam = GraphSettings.movingWindow.getFromOrDefault(source);
                expression = "moving_avg(" + expression + ", " + movingWindowParam + ")";
                break;
            }
            case "moving_percentile": {
                var movingWindowParam = GraphSettings.movingWindow.getFromOrDefault(source);
                var movingPercentileParam = GraphSettings.movingPercentile.getFromOrDefault(source);
                expression =
                        "moving_percentile(" + expression + ", " + movingWindowParam + ", " + movingPercentileParam + ")";
                break;
            }
            case "diff": {
                expression = "diff(" + expression + ")";
                break;
            }
            case "asap": {
                expression = "asap(" + expression + ")";
                break;
            }
            default:
                break;
        }

        if (!element.getTitle().isEmpty()) {
            String arg = Quoter.doubleQuote(element.getTitle());
            expression = "alias(" + expression + ", " + arg + ")";
        }

        Queries.Target target;
        if (isSelectorsTarget) {
            target = Queries.Target.newBuilder()
                    .setQuery(expression)
                    .build();
        } else {
            target = Queries.Target.newBuilder()
                    .setQuery(expression)
                    .setTextMode(true)
                    .build();
        }

        return target;
    }

    private static Selectors oldParametersToNewSelectors(Selector[] selectors) {
        SelectorsBuilder builder = Selectors.builder(selectors.length);

        for (var selector : selectors) {
            var name = selector.getName().trim();
            if (name.equals(LabelKeys.PROJECT)) {
                continue;
            }

            if (name.endsWith("!")) {
                builder.addOverride(ru.yandex.solomon.labels.query.Selector.notGlob(name.substring(0, name.length() - 1),
                        selector.getValue()));
            } else if (selector.getValue().startsWith("!")) {
                builder.addOverride(ru.yandex.solomon.labels.query.Selector.notGlob(name, selector.getValue().substring(1)));
            } else {
                builder.addOverride(ru.yandex.solomon.labels.query.Selector.glob(name, selector.getValue()));
            }
        }

        return builder.build();
    }

    static class ConvertedGraphData {
        public final ChartWidget.Builder inlineGraphWidgetBuilder;
        public final Parametrization parametrization;

        ConvertedGraphData(ChartWidget.Builder inlineGraphWidgetBuilder, Parametrization parametrization) {
            this.inlineGraphWidgetBuilder = inlineGraphWidgetBuilder;
            this.parametrization = parametrization;
        }
    }
}
