package ru.yandex.solomon.gateway.entityConverter;

import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.Doubles;
import com.google.common.primitives.Ints;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monitoring.v3.ChartWidget.NameHidingSettings;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings.GradientColorScheme;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.ColorSchemeSettings.StandardColorScheme;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.HeatmapSettings;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.SeriesAggregation;
import ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.VisualizationType;
import ru.yandex.monitoring.v3.Downsampling;
import ru.yandex.monitoring.v3.GapFilling;
import ru.yandex.monitoring.v3.GridAggregation;
import ru.yandex.solomon.gateway.backend.www.MetricsChecksSet;
import ru.yandex.solomon.gateway.utils.UserLinksBasic;
import ru.yandex.solomon.gateway.utils.conf.GraphSettings;
import ru.yandex.solomon.gateway.utils.conf.source.GraphSettingsSource;
import ru.yandex.solomon.math.Interpolate;
import ru.yandex.solomon.util.www.ser.GggDurationSerializer;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class GraphUtils {
    private static final Set<String> stackOffProjects = Set.of(
            "air", "bsyeti", "clickadd", "currency", "ichwill",
            "ichwill-dealer", "news");

    private static final Set<String> interpolateLeftProjects = Set.of("Antifraud", "clickadd", "genisys");
    private static final Set<String> interpolateNoneProjects = Set.of("yabs");

    static NameHidingSettings makeNameHidingSettings(String checks) {
        MetricsChecksSet metricsChecksSet = MetricsChecksSet.parse(checks);
        Set<String> ids = metricsChecksSet.getIds().stream()
                .map(id -> id.replaceAll(", ", "|"))
                .collect(Collectors.toSet());
        return NameHidingSettings.newBuilder()
                .setPositive(metricsChecksSet.isPositive())
                .addAllNames(ids)
                .build();
    }

    static VisualizationSettings makeVisualizationSettings(String projectParam, GraphSettingsSource source, @Nullable Boolean overrideStack) {
        var visualizationBuilder = VisualizationSettings.newBuilder();

        var graphModeParam = GraphSettings.graphMode.getFromOrDefault(source);
        var aggrParam = GraphSettings.aggr.getFromOrDefault(source);
        SeriesAggregation graphAggregation = convertGraphAggregation(aggrParam);

        switch (graphModeParam) {
            case UserLinksBasic.GRAPH_MODE_BARS: {
                visualizationBuilder
                        .setType(VisualizationType.VISUALIZATION_TYPE_BARS)
                        .setAggregation(graphAggregation);
                break;
            }
            case UserLinksBasic.GRAPH_MODE_PIE_CHART: {
                visualizationBuilder
                        .setShowLabels(true)
                        .setType(VisualizationType.VISUALIZATION_TYPE_PIE)
                        .setAggregation(graphAggregation);
                break;
            }
            case UserLinksBasic.GRAPH_MODE_DISTRIBUTION: {
                visualizationBuilder
                        .setType(VisualizationType.VISUALIZATION_TYPE_DISTRIBUTION)
                        .setAggregation(graphAggregation);
                break;
            }
            case UserLinksBasic.GRAPH_MODE_HEATMAP: {
                var greenValueParam = GraphSettings.greenValue.getFromOrDefault(source);
                var yellowValueParam = GraphSettings.yellowValue.getFromOrDefault(source);
                var redValueParam = GraphSettings.redValue.getFromOrDefault(source);
                var violetValueParam = GraphSettings.violetValue.getFromOrDefault(source);

                var heatmapSettings = HeatmapSettings.newBuilder()
                        .setGreenValue(greenValueParam)
                        .setYellowValue(yellowValueParam)
                        .setRedValue(redValueParam)
                        .setVioletValue(violetValueParam)
                        .build();

                visualizationBuilder
                        .setType(VisualizationType.VISUALIZATION_TYPE_HEATMAP)
                        .setAggregation(graphAggregation)
                        .setHeatmapSettings(heatmapSettings);
                break;
            }
            default: {
                boolean stack;

                String overLinesTransformFrom = GraphSettings.overLinesTransform.getFrom(source);
                if (!(overLinesTransformFrom.isEmpty() || overLinesTransformFrom.equals(GraphSettings.overLinesTransform.defaultValue))) {
                    stack = false;
                } else if (overrideStack != null) {
                    stack = overrideStack;
                } else {
                    String stackParam = GraphSettings.stack.getFrom(source);
                    if (!StringUtils.isEmpty(stackParam)) {
                        stack = GraphSettings.stack.getBoolFrom(source);
                    } else {
                        stack = !stackOffProjects.contains(projectParam);
                    }
                }

                var interpolateParam = GraphSettings.interpolate.getFrom(source);

                Interpolate oldInterpolate;
                if (interpolateParam.isEmpty()) {
                    if (interpolateLeftProjects.contains(projectParam)) {
                        oldInterpolate = Interpolate.LEFT;
                    } else if (interpolateNoneProjects.contains(projectParam)) {
                        oldInterpolate = Interpolate.NONE;
                    } else {
                        oldInterpolate = null;
                    }
                } else {
                    oldInterpolate = Interpolate.parse(interpolateParam);
                }

                ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate newInterpolate;
                if (oldInterpolate == null) {
                    newInterpolate = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate.INTERPOLATE_NOT_SPECIFIED;
                } else {
                    switch (oldInterpolate) {
                        case LINEAR:
                            newInterpolate = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate.INTERPOLATE_LINEAR;
                            break;
                        case LEFT:
                            newInterpolate = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate.INTERPOLATE_LEFT;
                            break;
                        case RIGHT:
                            newInterpolate = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate.INTERPOLATE_RIGHT;
                            break;
                        case NONE:
                            newInterpolate = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate.INTERPOLATE_NOT_SPECIFIED;
                            break;
                        default:
                            newInterpolate = ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate.INTERPOLATE_NOT_SPECIFIED;
                    }
                }

                if (oldInterpolate == Interpolate.NONE) {
                    if (stack) {
                        visualizationBuilder.setType(VisualizationType.VISUALIZATION_TYPE_COLUMN).setStack(true);
                    } else {
                        visualizationBuilder.setType(VisualizationType.VISUALIZATION_TYPE_POINTS);
                    }
                } else {
                    if (stack) {
                        visualizationBuilder.setType(VisualizationType.VISUALIZATION_TYPE_AREA).setStack(true);
                    } else {
                        visualizationBuilder.setType(VisualizationType.VISUALIZATION_TYPE_LINE);
                    }
                    if (newInterpolate != ru.yandex.monitoring.v3.ChartWidget.VisualizationSettings.Interpolate.INTERPOLATE_NOT_SPECIFIED) {
                        visualizationBuilder.setInterpolate(newInterpolate);
                    }
                }

                if (GraphSettings.norm.getBoolFrom(source)) {
                    visualizationBuilder.setNormalize(true);
                }
            }
        }

        var csParam = GraphSettings.cs.getFromOrDefault(source);

        ColorSchemeSettings colorSchemeSettings;
        switch (csParam) {
            case "default": {
                colorSchemeSettings = ColorSchemeSettings.newBuilder()
                        .setStandard(StandardColorScheme.getDefaultInstance())
                        .build();
                break;
            }
            case "gradient": {
                colorSchemeSettings = ColorSchemeSettings.newBuilder()
                        .setGradient(GradientColorScheme.newBuilder()
                                .setGreenValue(GraphSettings.green.getFromOrDefault(source))
                                .setYellowValue(GraphSettings.yellow.getFromOrDefault(source))
                                .setRedValue(GraphSettings.red.getFromOrDefault(source))
                                .setVioletValue(GraphSettings.violet.getFromOrDefault(source))
                                .build())
                        .build();
                break;
            }
            default: {
                // cs=auto equivalent.
                colorSchemeSettings = ColorSchemeSettings.getDefaultInstance();
            }
        }

        String yMinParam = GraphSettings.min.getFromOrDefault(source);
        String yMaxParam = GraphSettings.max.getFromOrDefault(source);
        String scale = GraphSettings.scale.getFromOrDefault(source);

        Double yMin = Doubles.tryParse(yMinParam);
        Double yMax = Doubles.tryParse(yMaxParam);

        boolean hasYMin = yMin != null && !yMin.isNaN();
        boolean hasYMax = yMax != null && !yMax.isNaN();

        if (hasYMin || hasYMax || !scale.isEmpty()) {
            VisualizationSettings.Yaxis.Builder yaxisBuilder = VisualizationSettings.Yaxis.newBuilder();
            if (hasYMin) {
                yaxisBuilder.setMin(yMinParam);
            }
            if (hasYMax) {
                yaxisBuilder.setMax(yMaxParam);
            }
            yaxisBuilder.setType(convertScaleType(scale));
            visualizationBuilder.setYaxisSettings(VisualizationSettings.YaxisSettings.newBuilder()
                    .setLeft(yaxisBuilder));
        }

        if (colorSchemeSettings.getSchemeCase() != ColorSchemeSettings.SchemeCase.SCHEME_NOT_SET) {
            visualizationBuilder.setColorSchemeSettings(colorSchemeSettings);
        }

        /*String thresholdParam = GraphSettings.threshold.getFromOrDefault(source);
        if (!thresholdParam.isEmpty()) {
            visualizationBuilder.addThresholds(Threshold.newBuilder()
                    .setValue(thresholdParam)
                    .setColor(ThresholdColor.THRESHOLD_COLOR_CRITICAL)
                    .build());
        }*/

        return visualizationBuilder.build();
    }

    public static Downsampling makeDownsampling(GraphSettingsSource source) {
        var downsamplingParam = GraphSettings.downsampling.getFromOrDefault(source);
        var downsamplingAggr = GraphSettings.downSamplingAggr.getFromOrDefault(source);
        var downsamplingFill = GraphSettings.downsamplingFill.getFrom(source);

        var downsamplingBuilder = Downsampling.newBuilder();

        switch (downsamplingParam) {
            case UserLinksBasic.DOWNSAMPLING_PARAM_AUTO: {
                break;
            }
            case UserLinksBasic.DOWNSAMPLING_PARAM_BY_INTERVAL: {
                var gridParam = GraphSettings.grid.getFromOrDefault(source);
                if (gridParam.isEmpty()) {
                    gridParam = GraphSettings.grid.hint;
                }

                var gridDuration = GggDurationSerializer.parseDuration(gridParam);
                int gridSeconds = gridDuration == null ? 60 : gridDuration.seconds;
                int gridMilliseconds = gridSeconds * 1000;

                downsamplingBuilder.setGridInterval(gridMilliseconds);
                break;
            }
            case UserLinksBasic.DOWNSAMPLING_PARAM_BY_POINTS: {
                var maxPointsParam = GraphSettings.maxPoints.getFromOrDefault(source);
                var maxPoints = Ints.tryParse(maxPointsParam);
                if (maxPoints == null) {
                    maxPoints = 100;
                }
                downsamplingBuilder.setMaxPoints(maxPoints);
                break;
            }
            case UserLinksBasic.DOWNSAMPLING_PARAM_OFF:
                downsamplingBuilder.setDisabled(true);
                break;
            default:
        }

        if (downsamplingBuilder.getModeCase() != Downsampling.ModeCase.DISABLED) {
            downsamplingBuilder.setGridAggregation(convertGridAggregation(downsamplingAggr));
            if (!downsamplingFill.isEmpty()) {
                downsamplingBuilder.setGapFilling(convertDownsamplingFill(downsamplingFill));
            }
        }
        return downsamplingBuilder.build();
    }

    private static GapFilling convertDownsamplingFill(String downsamplingFill) {
        switch (downsamplingFill.toLowerCase()) {
            case "null":
                return GapFilling.GAP_FILLING_NULL;
            case "none":
                return GapFilling.GAP_FILLING_NONE;
            case "previous":
                return GapFilling.GAP_FILLING_PREVIOUS;
            default:
                return GapFilling.GAP_FILLING_UNSPECIFIED;
        }
    }

    private static GridAggregation convertGridAggregation(String downsamplingAggr) {
        switch (downsamplingAggr.toLowerCase()) {
            case "avg":
                return GridAggregation.GRID_AGGREGATION_AVG;
            case "max":
                return GridAggregation.GRID_AGGREGATION_MAX;
            case "min":
                return GridAggregation.GRID_AGGREGATION_MIN;
            case "sum":
                return GridAggregation.GRID_AGGREGATION_SUM;
            case "last":
                return GridAggregation.GRID_AGGREGATION_LAST;
            default:
                return GridAggregation.GRID_AGGREGATION_UNSPECIFIED;
        }
    }

    private static SeriesAggregation convertGraphAggregation(String aggrParam) {
        switch (aggrParam.toLowerCase()) {
            case "avg":
                return SeriesAggregation.SERIES_AGGREGATION_AVG;
            case "max":
                return SeriesAggregation.SERIES_AGGREGATION_MAX;
            case "min":
                return SeriesAggregation.SERIES_AGGREGATION_MIN;
            case "sum":
                return SeriesAggregation.SERIES_AGGREGATION_SUM;
            case "last":
                return SeriesAggregation.SERIES_AGGREGATION_LAST;
            default:
                return SeriesAggregation.SERIES_AGGREGATION_NOT_SPECIFIED;
        }
    }

    private static VisualizationSettings.YaxisType convertScaleType(String scale) {
        switch (scale.toLowerCase()) {
            case "natural":
                return VisualizationSettings.YaxisType.YAXIS_TYPE_LINEAR;
            case "log":
                return VisualizationSettings.YaxisType.YAXIS_TYPE_LOGARITHMIC;
            default:
                return VisualizationSettings.YaxisType.YAXIS_TYPE_NOT_SPECIFIED;
        }
    }
}
