package ru.yandex.solomon.gateway.entityConverter;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.graph.Graph;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class LastVersions {

    private final Map<String, Integer> graphs;
    private final Map<String, Integer> dashboards;
    private final int projectMenuVersion;

    public LastVersions(
            Map<String, Integer> graphs,
            Map<String, Integer> dashboards,
            int projectMenuVersion)
    {
        this.graphs = graphs;
        this.dashboards = dashboards;
        this.projectMenuVersion = projectMenuVersion;
    }

    public Map<String, Integer> getDashboards() {
        return dashboards;
    }

    public Map<String, Integer> getGraphs() {
        return graphs;
    }

    public int getProjectMenuVersion() {
        return projectMenuVersion;
    }

    public static LastVersions create() {
        return new LastVersions(new HashMap<>(100), new HashMap<>(100), 0);
    }

    public static LastVersions fromEntities(Collection<Graph> graphs, Collection<Dashboard> dashboards, int projectMenuVersion) {
        var graphsMapping = graphs.stream()
                .collect(Collectors.toMap(Graph::getId, Graph::getVersion));
        var dashboardsMapping = dashboards.stream()
                .collect(Collectors.toMap(Dashboard::getId, Dashboard::getVersion));
        return new LastVersions(graphsMapping, dashboardsMapping, projectMenuVersion);
    }

    public static Diff createDiff(LastVersions prev, LastVersions next) {
        boolean changedProjectMenu = next.projectMenuVersion != prev.projectMenuVersion;
        Set<String> changedGraphIds = computeDiff(prev.graphs, next.graphs);
        Set<String> changedDashboardIds = computeDiff(prev.dashboards, next.dashboards);
        return new Diff(changedGraphIds, changedDashboardIds, changedProjectMenu);
    }

    private static Set<String> computeDiff(Map<String, Integer> prev, Map<String, Integer> next) {
        Set<String> result = new HashSet<>(100);
        for (var entry : next.entrySet()) {
            String id = entry.getKey();
            int version = entry.getValue();
            if (!prev.containsKey(id)) {
                result.add(id);
            } else {
                int prevVersion = prev.get(id);
                if (version != prevVersion) {
                    result.add(id);
                }
            }
        }
        return result;
    }

    public boolean isEmpty() {
        return graphs.isEmpty() && dashboards.isEmpty();
    }

    public static class Diff {
        final Set<String> changedGraphIds;
        final Set<String> changedDashboardIds;
        final boolean changedProjectMenu;

        public Diff(Set<String> changedGraphIds, Set<String> changedDashboardIds, boolean changedProjectMenu) {
            this.changedGraphIds = changedGraphIds;
            this.changedDashboardIds = changedDashboardIds;
            this.changedProjectMenu = changedProjectMenu;
        }

        public boolean hasDashboardsOrGraphs() {
            return changedGraphIds.isEmpty() && changedDashboardIds.isEmpty();
        }

        public boolean hasMenu() {
            return changedProjectMenu;
        }
    }
}
