package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.v3.ChartWidget;
import ru.yandex.monitoring.v3.QuickLinks;
import ru.yandex.monitoring.v3.QuickLinksData;
import ru.yandex.solomon.config.protobuf.frontend.EntityConverterConfig;
import ru.yandex.solomon.core.db.model.MenuItem;
import ru.yandex.solomon.core.db.model.ProjectMenu;
import ru.yandex.solomon.gateway.backend.page.WwwQueryArgsUtils;
import ru.yandex.solomon.gateway.utils.url.UrlUtils;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class QuickLinksConverter {
    private static final Logger logger = LoggerFactory.getLogger(QuickLinksConverter.class);

    public static CompletableFuture<QuickLinks> convertAsync(
            ProjectMenu projectMenu,
            EntityConverterConfig config,
            ExternalLoader externalLoader)
    {
        List<CompletableFuture<QuickLinksData.Item>> itemFutures = new ArrayList<>();

        for (var item : projectMenu.getItems()) {
            itemFutures.add(convertItemAsync(item, config, externalLoader));
        }

        return CompletableFutures.allOf(itemFutures).thenApply(rawItems -> {
            List<QuickLinksData.Item> items = rawItems.stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());

            QuickLinksData data = QuickLinksData.newBuilder()
                    .addAllItems(items)
                    .build();

            return QuickLinks.newBuilder()
                    .setData(data)
                    .setProjectId(projectMenu.getId())
                    .setCreatedBy(projectMenu.getCreatedBy())
                    .setUpdatedBy(projectMenu.getUpdatedBy())
                    .setCreatedAt(Timestamps.fromMillis(projectMenu.getUpdatedAtMillis()))
                    .setUpdatedAt(Timestamps.fromMillis(projectMenu.getCreatedAtMillis()))
                    .setVersion(projectMenu.getVersion())
                    .build();
        });
    }

    private static CompletableFuture<QuickLinksData.Item> convertItemAsync(
            MenuItem item,
            EntityConverterConfig config,
            ExternalLoader externalLoader)
    {
        if (item.getChildren().length > 0) {
            List<CompletableFuture<QuickLinksData.Item>> convertedSubItemFutures = new ArrayList<>();

            for (var subItem : item.getChildren()) {
                var convertedSubItemFuture = convertItemAsync(subItem, config, externalLoader);
                convertedSubItemFutures.add(convertedSubItemFuture);
            }

            return CompletableFutures.allOf(convertedSubItemFutures).thenApply(rawConvertedSubItems -> {
                List<QuickLinksData.Item> convertedSubItems = rawConvertedSubItems.stream()
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList());

                if (convertedSubItems.isEmpty()) {
                    return null;
                }

                var folderItem = QuickLinksData.FolderItem.newBuilder()
                        .setTitle(item.getTitle())
                        .addAllItems(convertedSubItems)
                        .build();

                return QuickLinksData.Item.newBuilder()
                        .setFolder(folderItem)
                        .build();
            });
        }

        boolean hasSelectors = !StringUtils.isEmpty(item.getSelectors());

        return convertUrlAsync(item.getUrl(), config, hasSelectors, externalLoader)
                .thenApply(newUrl -> {
                    if (newUrl == null) {
                        return null;
                    }

                    return QuickLinksData.Item.newBuilder()
                            .setLink(QuickLinksData.LinkItem.newBuilder()
                                    .setTitle(item.getTitle())
                                    .setUrl(newUrl)
                                    .build())
                            .build();
                });
    }

    private static CompletableFuture<String> convertUrlAsync(
            String url,
            EntityConverterConfig config,
            boolean hasSelectors,
            ExternalLoader externalLoader)
    {
        url = DashWidgetConverter.fixUrl(url, Map.of());

        if (url.isEmpty()) {
            // Empty panel.
            return CompletableFuture.completedFuture(null);
        }

        if (url.startsWith("/admin") || url.startsWith("admin")) {
            return CompletableFuture.completedFuture(config.getTargetSiteUrl() +"/" + StringUtils.removeStart(url, "/"));
        }

        if (url.startsWith("?")) {
            if (url.contains("&graph=auto&") || url.startsWith("?graph=auto") || url.endsWith("&graph=auto")) {
                if (hasSelectors) {
                    return CompletableFuture.completedFuture(null);
                }
                return CompletableFuture.completedFuture(autoGraphUrl(url));
            } else {
                Map<String, String> queryArgs = UrlUtils.parseUrlQueryArgs(url);
                Map<String, String> newQueryArgs = new LinkedHashMap<>();
                for (var entry : queryArgs.entrySet()) {
                    String name = entry.getKey();
                    String value = entry.getValue();
                    if (!(value.contains("{{") && value.contains("}}"))) {
                        newQueryArgs.put(name, value);
                    }
                }
                String newUrl = "?" + UrlUtils.makeQueryArgs(newQueryArgs);

                if (newUrl.contains("&graph=") || (newUrl.contains("?graph="))) {
                    return configuredGraphUrlAsync(newUrl, externalLoader);
                } else if (newUrl.contains("&dashboard=") || newUrl.contains("?dashboard=")) {
                    return dashboardUrlAsync(newUrl, externalLoader);
                }

                return CompletableFuture.completedFuture(null);
            }
        }

        // Add iframe
        return CompletableFuture.completedFuture(hasSelectors ? null : url);
    }

    @Nullable
    private static String autoGraphUrl(String url) {
        LinkedHashMap<String, String> params = new LinkedHashMap<>();

        String project = UrlUtils.parseUrlQueryArgs(url).get("project");
        if (project == null) {
            logger.error("Unknown autograph url: " + url);
            return null;
        }

        ChartWidget widget = AutoGraphWidgetConverter.createAutoGraphWidgetImpl("", url, "");

        if (widget == null) {
            return null;
        }

        List<ChartWidget.Queries.Target> targetsList = widget.getQueries().getTargetsList();

        for (int i = 0; i < targetsList.size(); i++) {
            ChartWidget.Queries.Target target = targetsList.get(i);
            String query = target.getQuery();
            boolean hidden = target.getHidden();
            boolean textMode = target.getTextMode();
            String name = "q." + i + "." + (textMode ? "text" : "s");
            params.put(name, query);
            if (hidden) {
                params.put("q." + i, "off");
            }
        }

        ChartWidget.VisualizationSettings visualizationSettings = widget.getVisualizationSettings();

        if (visualizationSettings.getType() == ChartWidget.VisualizationSettings.VisualizationType.VISUALIZATION_TYPE_AREA) {
            params.put("type", "area");
        } else if (visualizationSettings.getType() == ChartWidget.VisualizationSettings.VisualizationType.VISUALIZATION_TYPE_COLUMN) {
            params.put("type", "column");
        }

        if (visualizationSettings.getStack()) {
            params.put("stack", "on");
        }

        if (visualizationSettings.getNormalize()) {
            params.put("normz", "on");
        }

        return "/projects/" + project + "/explorer?" + UrlUtils.makeQueryArgs(params);
    }

    private static CompletableFuture<String> configuredGraphUrlAsync(String url, ExternalLoader externalLoader) {
        return graphOrDashboardUrlAsync(url, externalLoader, false);
    }

    private static CompletableFuture<String> dashboardUrlAsync(String url, ExternalLoader externalLoader) {
        return graphOrDashboardUrlAsync(url, externalLoader, true);
    }

    private static CompletableFuture<String> graphOrDashboardUrlAsync(String url, ExternalLoader externalLoader, boolean isDashboard) {
        LinkedHashMap<String, String> params = new LinkedHashMap<>();

        var urlParams = UrlUtils.parseUrlQueryArgs(url);
        String project = urlParams.get("project");
        String cluster = urlParams.getOrDefault("cluster", urlParams.getOrDefault("l.cluster", ""));
        String service = urlParams.getOrDefault("service", urlParams.getOrDefault("l.service", ""));
        if (!StringUtils.isEmpty(cluster)) {
            params.put("p.cluster", cluster);
        }
        if (!StringUtils.isEmpty(service)) {
            params.put("p.service", service);
        }

        return loadGeneratedId(isDashboard, project, urlParams, externalLoader)
                .thenApply(generatedId -> {
                    if (StringUtils.isEmpty(generatedId)) {
                        return null;
                    }

                    LinkedHashMap<String, String> labelParams =
                            WwwQueryArgsUtils.selectorsFromQuery(urlParams).toLinkedHashMap();

                    for (var entry : labelParams.entrySet()) {
                        params.put("p." + entry.getKey(), entry.getValue());
                    }

                    return "/projects/" + project + "/dashboards/" + generatedId + (params.isEmpty() ? "" :
                            "?" + UrlUtils.makeQueryArgs(params));
                });
    }

    private static CompletableFuture<String> loadGeneratedId(boolean isDashboard, String project, Map<String, String> urlParams, ExternalLoader externalLoader) {
        if (isDashboard) {
            String dashboardId = urlParams.get("dashboard");
            return externalLoader.loadDashboard(project, dashboardId)
                    .thenApply(dashboard -> dashboard == null ? "" : dashboard.getGeneratedId());
        } else {
            String graphId = urlParams.get("graph");
            return externalLoader.loadGraph(project, graphId)
                    .thenApply(graph -> graph == null ? "" : graph.getGeneratedId());
        }
    }
}
