package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.labels.selector.LabelSelectorSet;
import ru.yandex.solomon.util.parser.ParserSupport;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class SelectorsReplacer {
    private static final Pattern OLD_SELECTORS_REGEX = Pattern.compile("(load|load1)\\(([\"'])(.*?)([\"'])\\)", Pattern.MULTILINE);

    private static boolean isSelectorNameChar(char c) {
        return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') || c == '_' || c == '.';
    }

    private static int consumeBackQuotedName(String code, int pos, char quote) {
        var curPos = pos - 1;
        while (curPos >= 0) {
            var c = code.charAt(curPos);
            if (c == quote) {
                if (curPos > 0 && code.charAt(curPos - 1) == '\\') {
                    curPos -= 2;
                }
                break;
            }

            curPos -= 1;
        }

        return curPos + 1;
    }

    private static int consumeBackNameImpl(String code, int pos) {
        var curPos = pos;
        while (curPos >= 0) {
            var c = code.charAt(curPos);
            if (isSelectorNameChar(c)) {
                curPos -= 1;
            } else {
                break;
            }
        }

        if (curPos == pos) {
            return pos;
        }

        var name = code.substring(curPos + 1, pos + 1);

        if (name.equals("return")) {
            return pos;
        }

        return curPos + 1;
    }

    private static int consumeBackName(String code, int pos) {
        var curPos = pos - 1;

        while (curPos >= 0) {
            var c = code.charAt(curPos);
            if (Character.isWhitespace(c)) {
                curPos -= 1;
            } else {
                break;
            }
        }

        if (curPos < 0) {
            return pos;
        }

        var c = code.charAt(curPos);
        if (c == '"' || c == '\'') {
            return consumeBackQuotedName(code, curPos, c);
        }

        var startPos = consumeBackNameImpl(code, curPos);
        if (startPos == curPos) {
            return pos;
        }

        return startPos;
    }

    private static List<Part> divideToNewSelectorsInCode(String code) {
        var result = new ArrayList<Part>();

        var lastIndex = 0;
        var lastSearchIndex = 0;

        for (; ; ) {
            var pos = code.indexOf('{', lastSearchIndex);

            if (pos >= 0) {
                try {
                    var selectorsStartPos = consumeBackName(code, pos);
                    var parser = new ParserSupport(code, selectorsStartPos);
                    SelectorsFormat.consumeSelectors(parser, true);
                    var selectorsEndPos = parser.getPos();
                    if (selectorsEndPos == selectorsStartPos) {
                        // Go to next '{'
                        lastSearchIndex = pos + 1;
                    } else {
                        var textBefore = code.substring(lastIndex, selectorsStartPos);
                        if (!textBefore.isEmpty()) {
                            result.add(Part.text(textBefore));
                        }

                        var selectorsText = code.substring(selectorsStartPos, selectorsEndPos);
                        result.add(Part.selectors(selectorsText));
                        lastIndex = selectorsEndPos;
                        lastSearchIndex = lastIndex;
                    }
                } catch (Exception e) {
                    // Go to next "{"
                    lastSearchIndex = pos + 1;
                }
            } else {
                var text = code.substring(lastIndex);

                if (text.length() > 0) {
                    result.add(Part.text(text));
                }
                return result;
            }
        }
    }

    private static List<Part> divideToOldSelectorsInCode(String code) {
        var result = new ArrayList<Part>();

        var lastIndex = 0;

        var matcher = OLD_SELECTORS_REGEX.matcher(code);

        // eslint-disable-next-line no-cond-assign
        while (matcher.find()) {
            var matchedSubstr = matcher.group(0);
            var funcName = matcher.group(1);
            var selectors = matcher.group(3);
            var curIndex = matcher.start();

            var textBefore = code.substring(lastIndex, curIndex);
            if (!textBefore.isEmpty()) {
                result.add(Part.text(textBefore));
            }

            lastIndex = curIndex + matchedSubstr.length();
            if ("load".equals(funcName)) {
                result.add(Part.load(matchedSubstr, selectors));
            } else {
                result.add(Part.load1(matchedSubstr, selectors));
            }
        }

        var lastTextPart = code.substring(lastIndex);
        if (!lastTextPart.isEmpty()) {
            result.add(Part.text(lastTextPart));
        }

        return result;
    }

    /**
     * Divide code to parts with replaced selectors
     */
    public static List<Part> divideCodeBySelectors(String code) {
        try {
            var result = new ArrayList<Part>();

            var newSelectorsResult = divideToNewSelectorsInCode(code);

            for (Part part : newSelectorsResult) {
                if (part.type == PartType.TEXT) {
                    var subResult = divideToOldSelectorsInCode(part.text);
                    if (subResult.size() > 0) {
                        result.addAll(subResult);
                    }
                } else {
                    result.add(part);
                }
            }

            return result;
        } catch (Exception e) {
            return List.of(Part.text(code));
        }
    }

    private static Selectors mergeSelectorsAndParams(Selectors selectors, Map<String, String> params) {
        var builder = selectors.toBuilder();

        for (var entry : params.entrySet()) {
            var key = entry.getKey();
            var value = entry.getValue();
            if (value.isEmpty()) {
                value = "*";
            }
            if (!builder.hasKey(key)) {
                Selector selector;
                if ("*".equals(value)) {
                    selector = Selector.any(key);
                } else if ("-".equals(value)) {
                    selector = Selector.absent(key);
                } else if (value.startsWith("!")) {
                    selector = Selector.notGlob(key, value.substring(1));
                } else {
                    selector = Selector.glob(key, value);
                }
                builder.addOverride(selector);
            }
        }

        return builder.build();
    }

    private static String overrideSelectorsByParameters(PartType type, String selectors, Map<String, String> params) {
        Selectors parsedSelectors;
        if (type == PartType.SELECTORS) {
            parsedSelectors = Selectors.parse(selectors);
            var interpolatedLabelSelectors = parsedSelectors.stream()
                    .map(s -> {
                        var interpolatedValue = StringInterpolator.interpolatePattern(s.getValue(), params);
                        return Selector.of(s.getKey(), interpolatedValue, s.getType());
                    })
                    .toArray(Selector[]::new);
            parsedSelectors = Selectors.of(parsedSelectors.getNameSelector(), interpolatedLabelSelectors);
        } else {
            var interpolatedOldSelectors = StringInterpolator.interpolatePattern(selectors, params);
            parsedSelectors = LabelSelectorSet.parseEscaped(interpolatedOldSelectors).toNewSelectors();
        }

        var fixedSelectors = mergeSelectorsAndParams(parsedSelectors, params);

        var fixedSelectorsStr = Selectors.format(fixedSelectors);
        if (type == PartType.LOAD1) {
            fixedSelectorsStr = "single(" + fixedSelectorsStr + ")";
        }

        return fixedSelectorsStr;
    }

    public static String replaceSelectorsInCode(String code, Map<String, String> params) {
        var parts = divideCodeBySelectors(code);

        var replacedParts = new ArrayList<String>();

        for (Part part : parts) {
            if (part.type == PartType.TEXT) {
                replacedParts.add(part.text);
            } else {
                String replacementResult;
                if (part.type == PartType.SELECTORS) {
                    replacementResult = overrideSelectorsByParameters(part.type, part.text, params);
                } else {
                    replacementResult = overrideSelectorsByParameters(part.type, part.selectors, params);
                }
                replacedParts.add(replacementResult);
            }
        }

        return String.join("", replacedParts);
    }

    public enum PartType {
        SELECTORS,
        LOAD1,
        LOAD,
        TEXT,
    }

    public static class Part {
        public final PartType type;
        public final String text;
        // In case of load1, load.
        public final String selectors;

        public Part(PartType type, String text, String selectors) {
            this.type = type;
            this.text = text;
            this.selectors = selectors;
        }

        public static Part text(String text) {
            return new Part(PartType.TEXT, text, "");
        }

        public static Part selectors(String selectors) {
            return new Part(PartType.SELECTORS, selectors, "");
        }

        public static Part load(String text, String selectors) {
            return new Part(PartType.LOAD, text, selectors);
        }

        public static Part load1(String text, String selectors) {
            return new Part(PartType.LOAD1, text, selectors);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Part part = (Part) o;
            return type == part.type &&
                    text.equals(part.text) &&
                    selectors.equals(part.selectors);
        }

        @Override
        public int hashCode() {
            return Objects.hash(type, text, selectors);
        }

        @Override
        public String toString() {
            return "Part{" +
                    "type=" + type +
                    ", text='" + text + '\'' +
                    ", selectors='" + selectors + '\'' +
                    '}';
        }
    }
}
