package ru.yandex.solomon.gateway.entityConverter;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.solomon.util.jackson.SalmonJackson;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class SolomonApiClient {
    private final HttpClient httpClient;
    private final String host;
    private final String oauthToken;
    private static final ObjectMapper mapper = SalmonJackson.getObjectMapper();

    SolomonApiClient(String host, String oauthToken) {
        this.httpClient = HttpClient.newBuilder()
                .connectTimeout(Duration.ofSeconds(5))
                .build();
        this.host = host;
        this.oauthToken = oauthToken;
    }

    <T> CompletableFuture<T> loadRequest(Class<? extends T> clazz, String path, Map<String, String> params) {
        return loadRequest(path, params, content -> {
            try {
                return mapper.readValue(content, clazz);
            } catch (IOException e) {
                throw new IllegalStateException("failed to parse JSON \"" + content + "\"", e);
            }
        });
    }

    CompletableFuture<JsonNode> loadRequest(String path, Map<String, String> params) {
        return loadRequest(path, params, content -> {
            try {
                return mapper.readTree(content);
            } catch (IOException e) {
                throw new IllegalStateException("failed to parse JSON \"" + content + "\"", e);
            }
        });
    }

    private <T> CompletableFuture<T> loadRequest(String path, Map<String, String> params, Function<String, T> parseFunc) {
        URI uri = buildUri(path, params);

        HttpRequest httpRequest = HttpRequest.newBuilder()
                .GET()
                .uri(uri)
                .setHeader("Content-Type", "application/json")
                .setHeader("Authorization", "OAuth " + oauthToken)
                .build();

        try {
            return httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofString())
                    .thenApply(httpResponse -> {
                        String content = httpResponse.body();
                        if (httpResponse.statusCode() != 200) {
                            throw new IllegalStateException("failed to load request, path " + path + ", params: " + params + ", code:" +
                                    " " + httpResponse.statusCode() + ", content: " + content);
                        }

                        try {
                            return parseFunc.apply(content);
                        } catch (Exception e) {
                            throw new IllegalStateException("failed to parse JSON \"" + content + "\"", e);
                        }
                    });
        } catch (Exception e) {
            return CompletableFuture.failedFuture(new RuntimeException(e));
        }
    }

    public CompletableFuture<Void> postRequest(String path, Map<String, String> params, String body) {
        URI uri = buildUri(path, params);

        HttpRequest httpRequest = HttpRequest.newBuilder()
                .POST(HttpRequest.BodyPublishers.ofString(body))
                .uri(uri)
                .setHeader("Content-Type", "application/json")
                .setHeader("Authorization", "OAuth " + oauthToken)
                .build();

        try {
            return httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofString())
                    .thenAccept(httpResponse -> {
                        String content = httpResponse.body();
                        if (httpResponse.statusCode() != 200) {
                            throw new IllegalStateException("failed to post request, path " + path + ", params: " + params + ", code:" +
                                    " " + httpResponse.statusCode() + ", content: " + content);
                        }
                    });
        } catch (Exception e) {
            return CompletableFuture.failedFuture(new RuntimeException(e));
        }
    }

    private URI buildUri(String path, Map<String, String> params) {
        var uriBuilder = new UriBuilder()
                .setScheme("http")
                .setHost(host)
                .appendPath(path);
        params.forEach(uriBuilder::addParam);

        URI uri;
        try {
            uri = uriBuilder.build();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return uri;
    }
}
