package ru.yandex.solomon.gateway.entityConverter;

import java.util.ArrayList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;


@ParametersAreNonnullByDefault
public class TableWithSpansBuilder<T> {
    private ArrayList<ArrayList<Cell<T>>> rows;
    private int columnCount;

    TableWithSpansBuilder() {
        this.rows = new ArrayList<>();
        this.columnCount = 0;
    }

    @Nonnull
    Cell<T> getCellOrEmpty(int rowIndex, int columnIndex) {
        if (rowIndex >= 0 && rowIndex < rows.size()) {
            ArrayList<Cell<T>> row = rows.get(rowIndex);
            if (columnIndex >= 0 && columnIndex < row.size()) {
                return row.get(columnIndex);
            }
        }
        return Cell.empty();
    }

    void setSpannedCell(int rowIndex, int columnIndex, int rowspan, int colspan, T content) {
        colspan = Math.max(1, colspan);
        rowspan = Math.max(1, rowspan);
        for (int curColumnIndex = columnIndex; curColumnIndex < columnIndex + colspan; curColumnIndex++) {
            if (getCellOrEmpty(rowIndex, curColumnIndex).state() != CellState.EMPTY) {
                columnIndex = curColumnIndex + 1;
            }
        }
        for (int curRowIndex = rowIndex; curRowIndex < rowIndex + rowspan; ++curRowIndex) {
            for (int curColumnIndex = columnIndex; curColumnIndex < columnIndex + colspan; ++curColumnIndex) {
                setCell(curRowIndex, curColumnIndex, null, CellState.INSIDE_SPANNED_CELL, rowspan, colspan);
            }
        }
        setCell(rowIndex, columnIndex, content, CellState.FULL, rowspan, colspan);
    }

    private void setCell(int rowIndex, int columnIndex, @Nullable T content, CellState cellState, int rowspan, int colspan) {
        while (rowIndex >= rows.size()) {
            rows.add(new ArrayList<>());
        }
        ArrayList<Cell<T>> row = rows.get(rowIndex);
        while (columnIndex >= row.size()) {
            row.add(Cell.empty());
        }
        row.set(columnIndex, new Cell<>(content, cellState, rowspan, colspan));
        if (columnIndex >= columnCount) {
            columnCount = columnIndex + 1;
            for (ArrayList<Cell<T>> curRow : rows) {
                while (curRow.size() < columnCount) {
                    curRow.add(Cell.empty());
                }
            }
        }
    }
    int columnCount() {
        return columnCount;
    }
    int rowCount() {
        return rows.size();
    }
    static class Cell<T> {
        @Nullable
        private final T contentOrNull;
        private final CellState state;
        private final int rowspan;
        private final int colspan;
        Cell(@Nullable T contentOrNull, CellState state, int rowspan, int colspan) {
            this.contentOrNull = contentOrNull;
            this.state = state;
            this.rowspan = rowspan;
            this.colspan = colspan;
        }
        public static <T> Cell<T> empty() {
            return new Cell<T>(null, CellState.EMPTY, 1, 1);
        }
        public CellState state() {
            return state;
        }
        @Nullable
        T getContentOrNull() {
            return contentOrNull;
        }
        public int getRowspan() {
            return rowspan;
        }
        public int getColspan() {
            return colspan;
        }
        boolean isEmptyCell() {
            return state != CellState.FULL;
        }
    }
    enum CellState {
        EMPTY,
        FULL,
        INSIDE_SPANNED_CELL,
    }
}
