package ru.yandex.solomon.gateway.filters;

import java.util.Optional;

import io.opentracing.Scope;
import io.opentracing.Span;
import io.opentracing.SpanContext;
import io.opentracing.Tracer;
import io.opentracing.noop.NoopSpan;
import io.opentracing.propagation.Format;
import io.opentracing.tag.Tags;
import org.springframework.context.annotation.Import;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;
import reactor.core.publisher.SignalType;

import ru.yandex.monlib.metrics.webflux.filters.RequestPatterns;

/**
 * @author Sergey Polovko
 */
@Component
@Import(RequestPatterns.class)
public class TracingFilter implements WebFilter {

    private final Tracer tracer;
    private final RequestPatterns patterns;

    public TracingFilter(Tracer tracer, RequestPatterns patterns) {
        this.tracer = tracer;
        this.patterns = patterns;
    }

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, WebFilterChain chain) {
        Optional<String> patternOpt = patterns.findRequestPattern(exchange.getRequest());

        final Span span;
        if (patternOpt.isPresent()) {
            ServerHttpRequest request = exchange.getRequest();
            SpanContext parentSpan = tracer.extract(
                    Format.Builtin.HTTP_HEADERS,
                    new HttpHeadersAdapter(request.getHeaders()));

            span = tracer.buildSpan(patternOpt.get())
                    .asChildOf(parentSpan)
                    .withTag(Tags.COMPONENT, "http")
                    .withTag(Tags.SPAN_KIND, Tags.SPAN_KIND_SERVER)
                    .withTag(Tags.HTTP_METHOD, request.getMethod().name())
                    .start();
        } else {
            span = NoopSpan.INSTANCE;
        }

        try (Scope scope = tracer.activateSpan(span)) {
            return chain.filter(exchange).doFinally(signalType -> {
                span.setTag(Tags.HTTP_STATUS, exchange.getResponse().getRawStatusCode());
                if (signalType == SignalType.ON_ERROR) {
                    span.setTag(Tags.ERROR, Boolean.TRUE);
                }
                span.finish();
            });
        }
    }
}
