package ru.yandex.solomon.gateway.operations.db.ydb;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AutoPartitioningPolicy;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.PartitioningPolicy;
import com.yandex.ydb.table.values.PrimitiveType;

import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.LongRunningOperationType;
import ru.yandex.solomon.gateway.operations.db.LongRunningOperationDao;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.solomon.ydb.YdbTable;
import ru.yandex.solomon.ydb.page.PageTokenCodec;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.string;
import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class YdbLongRunningOperationDao implements LongRunningOperationDao {

    private final Table table;
    private final YdbLongRunningOperationQuery query;

    public YdbLongRunningOperationDao(String root, TableClient tableClient) {
        var tablePath = root + "/LongRunningOperations";
        this.table = new Table(tableClient, tablePath);
        this.query = new YdbLongRunningOperationQuery(tablePath, Table.LIST_INDEX);
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return table.create();
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        return table.drop();
    }

    @Override
    public CompletableFuture<Boolean> insert(LongRunningOperation operation) {
        return table.insertOne(query.insert, operation);
    }

    @Override
    public CompletableFuture<Optional<LongRunningOperation>> insertIfAbsent(LongRunningOperation operation) {
        return safeCall(() -> table.queryOne(query.insertIfAbsent, table.toParams(operation)));
    }

    @Override
    public CompletableFuture<Optional<LongRunningOperation>> findOne(String operationId) {
        return safeCall(() -> {
            var params = Params.create()
                .put("$operationId", utf8(operationId));

            return table.queryOne(query.find, params);
        });
    }

    @Override
    public CompletableFuture<Optional<LongRunningOperation>> update(LongRunningOperation operation) {
        return table.updateOne(query.update, operation);
    }

    @Override
    public CompletableFuture<TokenBasePage<LongRunningOperation>> list(
        ContainerType containerType,
        String containerId,
        LongRunningOperationType operationType,
        TokenPageOptions pageOpts)
    {
        return safeCall(() -> {
            var pageToken = PageToken.decode(pageOpts.getPageToken());
            var params = Params.create()
                .put("$containerType", utf8(containerType.name()))
                .put("$containerId", utf8(containerId))
                .put("$operationType", utf8(operationType.name()))
                .put("$lastCreatedAt", timestamp(TimeUnit.MILLISECONDS.toMicros(pageToken.lastCreatedAt)))
                .put("$lastOperationId", utf8(pageToken.lastOperationId))
                .put("$pageSize", int32(pageOpts.getSize() + 1));

            return table.queryPage(
                query.list,
                params,
                pageOpts,
                op -> new PageToken(op.createdAt(), op.operationId()).encode());
        });
    }

    @Override
    public CompletableFuture<Long> count(
        ContainerType containerType,
        String containerId,
        LongRunningOperationType operationType,
        long createdSince,
        int limit)
    {
        return safeCall(() -> {
            var params = Params.create()
                .put("$containerType", utf8(containerType.name()))
                .put("$containerId", utf8(containerId))
                .put("$operationType", utf8(operationType.name()))
                .put("$createdSince", timestamp(TimeUnit.MILLISECONDS.toMicros(createdSince)))
                .put("$limit", int32(limit));

            return table.executeAndExpectSuccess(query.count, params)
                .thenApply(result -> {
                    var rs = result.getResultSet(0);
                    if (!rs.next()) {
                        return 0L;
                    }

                    return rs.getColumn(0).getUint64();
                });
        });
    }

    private static class Table extends YdbTable<String, LongRunningOperation> {

        static final String LIST_INDEX = "listIndex";

        Table(TableClient tableClient, String path) {
            super(tableClient, path);
        }

        @Override
        protected TableDescription description() {
            return TableDescription.newBuilder()
                .addNullableColumn("operationId", PrimitiveType.utf8())
                .addNullableColumn("operationType", PrimitiveType.utf8())
                .addNullableColumn("containerId", PrimitiveType.utf8())
                .addNullableColumn("containerType", PrimitiveType.utf8())
                .addNullableColumn("description", PrimitiveType.utf8())
                .addNullableColumn("createdAt", PrimitiveType.timestamp())
                .addNullableColumn("createdBy", PrimitiveType.utf8())
                .addNullableColumn("updatedAt", PrimitiveType.timestamp())
                .addNullableColumn("status", PrimitiveType.int32())
                .addNullableColumn("data", PrimitiveType.string())
                .addNullableColumn("version", PrimitiveType.int32())
                .setPrimaryKeys("operationId")
                .addGlobalIndex(LIST_INDEX, List.of("containerType", "containerId", "operationType", "createdAt"))
                .build();
        }

        @Override
        protected CreateTableSettings createTableSettings(CreateTableSettings settings) {
            var partitioningPolicy = new PartitioningPolicy()
                .setAutoPartitioning(AutoPartitioningPolicy.AUTO_SPLIT_MERGE);
            return settings.setPartitioningPolicy(partitioningPolicy);
        }

        @Override
        protected String getId(LongRunningOperation operation) {
            return operation.operationId();
        }

        @Override
        protected Params toParams(LongRunningOperation operation) {
            return Params.create()
                .put("$operationId", utf8(operation.operationId()))
                .put("$operationType", utf8(operation.operationType().name()))
                .put("$containerId", utf8(operation.containerId()))
                .put("$containerType", utf8(operation.containerType().name()))
                .put("$description", utf8(operation.description()))
                .put("$createdAt", timestamp(TimeUnit.MILLISECONDS.toMicros(operation.createdAt())))
                .put("$createdBy", utf8(operation.createdBy()))
                .put("$updatedAt", timestamp(TimeUnit.MILLISECONDS.toMicros(operation.updatedAt())))
                .put("$status", int32(operation.status()))
                .put("$data", string(operation.data().toByteString()))
                .put("$version", int32(operation.version()));
        }

        @Override
        protected LongRunningOperation mapFull(ResultSetReader resultSet) {
            return LongRunningOperation.newBuilder()
                .setOperationId(resultSet.getColumn("operationId").getUtf8())
                .setOperationType(LongRunningOperationType.valueOf(resultSet.getColumn("operationType").getUtf8()))
                .setContainerId(resultSet.getColumn("containerId").getUtf8())
                .setContainerType(ContainerType.valueOf(resultSet.getColumn("containerType").getUtf8()))
                .setDescription(resultSet.getColumn("description").getUtf8())
                .setCreatedAt(resultSet.getColumn("createdAt").getTimestamp().toEpochMilli())
                .setCreatedBy(resultSet.getColumn("createdBy").getUtf8())
                .setUpdatedAt(resultSet.getColumn("updatedAt").getTimestamp().toEpochMilli())
                .setStatus(resultSet.getColumn("status").getInt32())
                .setData(any(resultSet, "data"))
                .setVersion(resultSet.getColumn("version").getInt32())
                .build();
        }

        @Override
        protected LongRunningOperation mapPartial(ResultSetReader resultSet) {
            return mapFull(resultSet);
        }

        private static Any any(ResultSetReader rs, String name) {
            try {
                return Any.parseFrom(rs.getColumn(name).getString());
            } catch (InvalidProtocolBufferException ex) {
                throw new RuntimeException("Unable to parse column " + name + " at row " + rs.getRowCount());
            }
        }
    }

    @ParametersAreNonnullByDefault
    private static record PageToken(
        @JsonProperty("lastCreatedAt") long lastCreatedAt,
        @JsonProperty("lastOperationId") String lastOperationId)
    {
        private static final PageToken EMPTY = new PageToken(InstantUtils.NOT_AFTER, "");
        private static final PageTokenCodec<PageToken> codec = PageTokenCodec.forType(PageToken.class);

        static PageToken decode(CharSequence token) {
            return token.isEmpty() ? EMPTY : codec.decode(token);
        }

        String encode() {
            return codec.encode(this);
        }
    }
}
