package ru.yandex.solomon.gateway.operations.db.ydb;

import javax.annotation.ParametersAreNonnullByDefault;

import org.intellij.lang.annotations.Language;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
class YdbLongRunningOperationQuery {

    @Language("SQL")
    private static final String FIND = """
            --!syntax_v1
            DECLARE $operationId AS Utf8;

            SELECT * FROM `TABLE_PATH`
            WHERE operationId = $operationId;
            """;

    @Language("SQL")
    private static final String INSERT = """
            --!syntax_v1
            DECLARE $operationId AS Utf8;
            DECLARE $operationType AS Utf8;
            DECLARE $containerId AS Utf8;
            DECLARE $containerType AS Utf8;
            DECLARE $description AS Utf8;
            DECLARE $createdAt AS Timestamp;
            DECLARE $createdBy AS Utf8;
            DECLARE $updatedAt AS Timestamp;
            DECLARE $status AS Int32;
            DECLARE $data AS String;
            DECLARE $version AS Int32;

            INSERT INTO `TABLE_PATH` (
              operationId,
              operationType,
              containerId,
              containerType,
              description,
              createdAt,
              createdBy,
              updatedAt,
              status,
              data,
              version
            )
            VALUES(
              $operationId,
              $operationType,
              $containerId,
              $containerType,
              $description,
              $createdAt,
              $createdBy,
              $updatedAt,
              $status,
              $data,
              $version
            );
            """;

    @Language("SQL")
    private static final String INSERT_IF_ABSENT = """
            --!syntax_v1
            DECLARE $operationId AS Utf8;
            DECLARE $operationType AS Utf8;
            DECLARE $containerId AS Utf8;
            DECLARE $containerType AS Utf8;
            DECLARE $description AS Utf8;
            DECLARE $createdAt AS Timestamp;
            DECLARE $createdBy AS Utf8;
            DECLARE $updatedAt AS Timestamp;
            DECLARE $status AS Int32;
            DECLARE $data AS String;
            DECLARE $version AS Int32;

            $new = (
              SELECT
                $operationId as operationId,
                $operationType as operationType,
                $containerId as containerId,
                $containerType as containerType,
                $description as description,
                $createdAt as createdAt,
                $createdBy as createdBy,
                $updatedAt as updatedAt,
                $status as status,
                $data as data,
                $version as version
            );

            $old = (
              SELECT * FROM `TABLE_PATH`
              WHERE operationId = $operationId
            );

            INSERT INTO `TABLE_PATH`
            SELECT new.*
            FROM
              $new AS new
              LEFT JOIN $old AS old
              USING (operationId)
            WHERE old.operationId IS NULL;

            SELECT * FROM $old;
            """;

    @Language("SQL")
    private static final String LIST = """
            --!syntax_v1
            DECLARE $containerType AS Utf8;
            DECLARE $containerId AS Utf8;
            DECLARE $operationType AS Utf8;
            DECLARE $lastCreatedAt AS Timestamp;
            DECLARE $lastOperationId AS Utf8;
            DECLARE $pageSize AS Int32;

            $keys = (
                SELECT operationId, createdAt
                FROM `TABLE_PATH` VIEW `LIST_INDEX`
                WHERE containerType = $containerType
                  AND containerId = $containerId
                  AND operationType = $operationType
                  AND (createdAt = $lastCreatedAt AND operationId > $lastOperationId OR createdAt < $lastCreatedAt)
                ORDER BY createdAt DESC
                LIMIT $pageSize
            );

            SELECT t.*
            FROM $keys AS k
            INNER JOIN `TABLE_PATH` AS t ON (t.operationId = k.operationId);
            """;

    @Language("SQL")
    private static final String UPDATE = """
            --!syntax_v1
            DECLARE $operationId AS Utf8;
            DECLARE $operationType AS Utf8;
            DECLARE $containerId AS Utf8;
            DECLARE $containerType AS Utf8;
            DECLARE $description AS Utf8;
            DECLARE $updatedAt AS Timestamp;
            DECLARE $status AS Int32;
            DECLARE $data AS String;
            DECLARE $version AS Int32;

            $update = (
              SELECT
                operationId,
                operationType,
                containerId,
                containerType,
                $description AS description,
                createdAt,
                createdBy,
                $updatedAt AS updatedAt,
                $status AS status,
                $data AS data,
                version + 1 AS version
              FROM `TABLE_PATH`
              WHERE operationId = $operationId
                AND version = $version
            );

            SELECT * FROM $update;

            UPSERT INTO `TABLE_PATH`
            SELECT * FROM $update;
            """;

    @Language("SQL")
    private static final String COUNT = """
            --!syntax_v1
            DECLARE $containerType AS Utf8;
            DECLARE $containerId AS Utf8;
            DECLARE $operationType AS Utf8;
            DECLARE $createdSince AS Timestamp;
            DECLARE $limit AS Int32;

            $lastOps = (
                SELECT operationId
                FROM `TABLE_PATH` VIEW `LIST_INDEX`
                WHERE containerType = $containerType
                    AND containerId = $containerId
                    AND operationType = $operationType
                    AND createdAt > $createdSince
                LIMIT $limit
            );

            SELECT count(*)
            FROM $lastOps;
            """;

    private final String tablePath;
    private final String listIndex;

    final String find;
    final String insert;
    final String insertIfAbsent;
    final String list;
    final String update;
    final String count;

    YdbLongRunningOperationQuery(String tablePath, String listIndex) {
        this.tablePath = tablePath;
        this.listIndex = listIndex;

        this.find = prepare(FIND);
        this.insert = prepare(INSERT);
        this.insertIfAbsent = prepare(INSERT_IF_ABSENT);
        this.list = prepare(LIST);
        this.update = prepare(UPDATE);
        this.count = prepare(COUNT);
    }

    private String prepare(String query) {
        return query
            .replaceAll("TABLE_PATH", tablePath)
            .replaceAll("LIST_INDEX", listIndex);
    }
}
