package ru.yandex.solomon.gateway.operations.deleteMetrics;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.gateway.operations.DeleteMetricsOperationData;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.LongRunningOperationType;
import ru.yandex.solomon.gateway.operations.db.LongRunningOperationDao;
import ru.yandex.solomon.util.Proto;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.ydb.page.TokenBasePage;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class DeleteMetricsOperationManager {

    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
        .withNumRetries(10)
        .withDelay(TimeUnit.MILLISECONDS.toMillis(1))
        .withMaxDelay(TimeUnit.MILLISECONDS.toMillis(50));

    private static final long PROJECT_LIMIT_WINDOW_MILLIS = TimeUnit.DAYS.toMillis(1);
    public static final int PROJECT_LIMIT = 1000;

    private final LongRunningOperationDao dao;

    public static DeleteMetricsOperationData unpackData(LongRunningOperation operation) {
        return Proto.unpack(operation.data(), DeleteMetricsOperationData.getDefaultInstance());
    }

    public DeleteMetricsOperationManager(LongRunningOperationDao dao) {
        this.dao = dao;
    }

    public CompletableFuture<CreateResult> tryCreateOperation(
        String operationId,
        DeleteMetricsParams params)
    {
        var operation = makeDeleteMetricsOperation(operationId, params);
        return dao.insertIfAbsent(operation)
            .thenApply(
                existing -> existing
                    .map(op -> new CreateResult(op, Status.ALREADY_EXISTS))
                    .orElseGet(() -> new CreateResult(operation, Status.OK)))
            .exceptionally(
                t -> new CreateResult(
                    operation,
                    Status.UNKNOWN.withDescription("failed to insert operation").withCause(t)));
    }

    public CompletableFuture<Optional<LongRunningOperation>> getOperation(String operationId) {
        return dao.findOne(operationId);
    }

    public CompletableFuture<LongRunningOperation> getOperationOrThrow(String operationId) {
        return getOperation(operationId)
            .thenApply(op -> op.orElseThrow(() -> operationNotFound(operationId)));
    }

    public CompletableFuture<TokenBasePage<LongRunningOperation>> listOperations(
        String projectId,
        TokenPageOptions pageOpts)
    {
        return dao.list(
            ContainerType.PROJECT,
            projectId,
            LongRunningOperationType.DELETE_METRICS,
            pageOpts);
    }

    public CompletableFuture<Optional<LongRunningOperation>> updateOperation(LongRunningOperation operation) {
        var operationType = operation.operationType();
        checkArgument(operationType == LongRunningOperationType.DELETE_METRICS, "unexpected: %s", operationType);
        var containerType = operation.containerType();
        checkArgument(containerType == ContainerType.PROJECT, "unexpected: %s", containerType);

        return dao.update(operation);
    }

    public CompletableFuture<LongRunningOperation> forceUpdateOperation(LongRunningOperation operation) {
        return updateOperation(operation)
            .thenCompose(
                updated -> updated
                    .map(CompletableFuture::completedFuture)
                    .orElseGet(() -> runWithRetries(() -> tryForceUpdateOperation(operation), RETRY_CONFIG)));
    }

    public CompletableFuture<Boolean> checkProjectLimitReached(String projectId, long now) {
        return dao.count(
                ContainerType.PROJECT,
                projectId,
                LongRunningOperationType.DELETE_METRICS,
                now - PROJECT_LIMIT_WINDOW_MILLIS,
                PROJECT_LIMIT)
            .thenApply(count -> count >= PROJECT_LIMIT);
    }

    private CompletableFuture<LongRunningOperation> tryForceUpdateOperation(LongRunningOperation operation) {
        return getOperation(operation.operationId())
            .thenCompose(prev -> {
                var update = operation.toBuilder()
                    .setVersion(prev.orElseThrow().version())
                    .build();

                return updateOperation(update).thenApply(
                    updated -> updated.orElseThrow(
                        () -> Status.ABORTED
                            .withDescription("unable to force update")
                            .asRuntimeException()));
            });
    }

    private static LongRunningOperation makeDeleteMetricsOperation(String operationId, DeleteMetricsParams params) {
        var data = DeleteMetricsOperationData.newBuilder()
            .setSelectors(params.getSelectors())
            .setPermanentDeletionAt(params.getPermanentDeletionAt())
            .build();

        return LongRunningOperation.newBuilder()
            .setOperationId(operationId)
            .setOperationType(LongRunningOperationType.DELETE_METRICS)
            .setContainerId(params.getProjectId())
            .setContainerType(ContainerType.PROJECT)
            .setDescription(params.getDescription())
            .setCreatedAt(params.getCreatedAt())
            .setCreatedBy(params.getCreatedBy())
            .setUpdatedAt(params.getCreatedAt())
            .setStatus(DeleteMetricsOperationStatus.DELETING.value)
            .setData(Any.pack(data))
            .setVersion(1)
            .build();
    }

    private static StatusRuntimeException operationNotFound(String operationId) {
        return Status.NOT_FOUND
            .withDescription("operation not found: " + operationId)
            .asRuntimeException();
    }

    public record CreateResult(LongRunningOperation operation, Status status) {
        public boolean isOk() {
            return status.isOk();
        }

        public boolean isAlreadyExists() {
            return status().getCode() == Status.Code.ALREADY_EXISTS;
        }
    }
}
