package ru.yandex.solomon.gateway.operations.deleteMetrics;

import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.frontend.DeleteMetricsConfig;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class DeleteMetricsOperationMetrics {

    private final ConcurrentMap<MetricsKey, Metrics> metrics = new ConcurrentHashMap<>();

    private final MetricRegistry registry;
    private final boolean verbose;

    public DeleteMetricsOperationMetrics(MetricRegistry registry, DeleteMetricsConfig config) {
        this.registry = registry;
        this.verbose = config.getReportVerboseMetrics();

        prepareMetrics("total", "operation");
    }

    public void touch(String projectId) {
        prepareMetrics(projectId, "total");
    }

    public void touch(String projectId, String shardId) {
        prepareMetrics(projectId, shardId);
    }

    public void operationStarted(String projectId) {
        incOperationStarted("total", "operation");
        incOperationStarted(projectId, "operation");
    }

    public void operationStarted(String projectId, String shardId) {
        if (verbose) {
            incOperationStarted(projectId, shardId);
        }
    }

    public void operationCancelled(String projectId) {
        incOperationCancelled("total", "operation");
        incOperationCancelled(projectId, "operation");
    }

    public void operationCancelled(String projectId, String shardId) {
        if (verbose) {
            incOperationCancelled(projectId, shardId);
        }
    }

    public void operationCancelFailed(String projectId) {
        incOperationCancelFailed("total");
        incOperationCancelFailed(projectId);
    }

    public void moveCompleteOnReplicas(String projectId, int elapsedSeconds) {
        recordMoveCompleteElapsedSeconds("total", elapsedSeconds);
        recordMoveCompleteElapsedSeconds(projectId, elapsedSeconds);
    }

    public void operationTerminated(String projectId, DeleteMetricsOperationStatus status) {
        incOperationTerminated("total", "operation", status);
        incOperationTerminated(projectId, "operation", status);
    }

    public void operationTerminated(String projectId, String shardId, DeleteMetricsOperationStatus status) {
        if (verbose) {
            incOperationTerminated(projectId, shardId, status);
        }
    }

    private void incOperationStarted(String projectId, String shardId) {
        prepareMetrics(projectId, shardId).started.inc();
    }

    private void incOperationCancelled(String projectId, String shardId) {
        prepareMetrics(projectId, shardId).cancelled.inc();
    }

    private void incOperationCancelFailed(String projectId) {
        prepareMetrics(projectId, "operation").cancelFailed.inc();
    }

    private void recordMoveCompleteElapsedSeconds(String projectId, int elapsedSeconds) {
        prepareMetrics(projectId, "operation").moveCompleteElapsedSeconds.record(elapsedSeconds);
    }

    private void incOperationTerminated(String projectId, String shardId, DeleteMetricsOperationStatus status) {
        assert status.isTerminal();
        var rate = prepareMetrics(projectId, shardId).terminated.get(status);
        if (rate != null) {
            rate.inc();
        }
    }

    private Metrics prepareMetrics(String projectId, String shardId) {
        var key = new MetricsKey(projectId, shardId);
        var result = metrics.get(key);
        if (result == null) {
            metrics.putIfAbsent(key, result = new Metrics(key, registry));
        }
        return result;
    }

    private record MetricsKey(String projectId, String shardId) { }

    private static final class Metrics {
        final Rate started;
        final Rate cancelled;
        final Rate cancelFailed;
        final Histogram moveCompleteElapsedSeconds;
        final Map<DeleteMetricsOperationStatus, Rate> terminated;

        Metrics(MetricsKey key, MetricRegistry registry) {
            var labels = Labels.of("projectId", key.projectId(), "shardId", key.shardId());
            var subRegistry = registry.subRegistry(labels);

            this.started = subRegistry.rate("delete_metrics.operation.started");
            this.cancelled = subRegistry.rate("delete_metrics.operation.cancelled");
            this.cancelFailed = subRegistry.rate("delete_metrics.operation.cancel_failed");
            this.moveCompleteElapsedSeconds = subRegistry.histogramRate(
                "delete_metrics.operation.move_complete_elapsed_seconds",
                Histograms.exponential(13, 2, 64));

            var terminated = new EnumMap<DeleteMetricsOperationStatus, Rate>(DeleteMetricsOperationStatus.class);
            for (var status : DeleteMetricsOperationStatus.terminal()) {
                terminated.put(
                    status,
                    subRegistry.rate(
                        "delete_metrics.operation.terminated",
                        Labels.of("status", status.name())));
            }
            this.terminated = terminated;
        }
    }
}
