package ru.yandex.solomon.gateway.operations.deleteMetrics;

import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.ydb.page.TokenPageOptions;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager.unpackData;
import static ru.yandex.solomon.staffOnly.annotations.ManagerMethod.ExecuteMethod.POST;

/**
 * @author Stanislav Kashirin
 */
//TODO: this is one-off patch, should be removed after usage
@ParametersAreNonnullByDefault
final class DeleteMetricsOperationTrackerHistoryPatch {

    private static final Set<String> PROJECTS = Set.of(
        "direct",
        "distbuild",
        "geoadv",
        "inventori",
        "market-checkout",
        "market-tpl",
        "release_machine",
        "taxi",
        "telephony",
        "webmaster",
        "yasm_antirobot",
        "yasm_mdsproxy",
        "yasm_vhfrontend",
        "yasm_yabs_hit_models",
        "ydo_frontend_tools_monitoring");

    private static final long OLD_OPERATIONS_THRESHOLD =
        LocalDate.of(2022, 2, 24)
            .atStartOfDay()
            .toInstant(ZoneOffset.UTC)
            .toEpochMilli();

    private final DeleteMetricsOperationManager manager;
    private final DeleteMetricsOperationTracker tracker;

    DeleteMetricsOperationTrackerHistoryPatch(
        DeleteMetricsOperationManager manager,
        DeleteMetricsOperationTracker tracker)
    {
        this.manager = manager;
        this.tracker = tracker;
    }

    @ManagerMethod(executeMethod = POST)
    CompletableFuture<Void> patchHistory() {
        return listOperations()
            .thenAccept(ops -> {
                for (var op : ops) {
                    var data = unpackData(op);

                    if (op.createdAt() <= OLD_OPERATIONS_THRESHOLD) {
                        tracker.operationStarted(
                            op.createdAt(),
                            op.createdBy(),
                            op.containerType(),
                            op.containerId(),
                            op.operationId(),
                            data.getSelectors()
                        );
                    }

                    var status = DeleteMetricsOperationStatus.fromValue(op.status());
                    if (status.isTerminal()) {
                        tracker.operationTerminated(
                            op.updatedAt(),
                            ContainerType.PROJECT,
                            op.containerId(),
                            op.operationId(),
                            status,
                            data.getStatusMessage(),
                            data.getPermanentlyDeletedMetricsCount()
                        );
                    }
                }
            });
    }

    CompletableFuture<List<LongRunningOperation>> listOperations() {
        var listProjects = PROJECTS.stream()
            .map(projectId -> listOperations(new ArrayList<>(), projectId, ""))
            .collect(collectingAndThen(toList(), CompletableFutures::allOf));

        return listProjects.thenApply(CollectionF::flatten);
    }

    CompletableFuture<List<LongRunningOperation>> listOperations(
        List<LongRunningOperation> ops,
        String projectId,
        String token)
    {
        return manager.listOperations(projectId, new TokenPageOptions(100, token))
            .thenCompose(page -> {
                ops.addAll(page.getItems());

                var nextToken = page.getNextPageToken();
                if (nextToken.isEmpty()) {
                    return completedFuture(ops);
                }

                return listOperations(ops, projectId, nextToken);
            });
    }

}
