package ru.yandex.solomon.gateway.operations.deleteMetrics;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.logbroker.agent.client.Session;
import ru.yandex.solomon.core.container.ContainerType;

import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class DeleteMetricsOperationTrackerImpl implements DeleteMetricsOperationTracker {

    private static final Logger logger = LoggerFactory.getLogger(DeleteMetricsOperationTrackerImpl.class);

    @WillCloseWhenClosed
    private final Session session;
    private final ObjectMapper mapper;

    public DeleteMetricsOperationTrackerImpl(@WillCloseWhenClosed Session session) {
        this.session = session;
        this.mapper = new ObjectMapper();
    }

    @Override
    public void operationStarted(
        long tsMillis,
        String login,
        ContainerType containerType,
        String containerId,
        String operationId,
        String selectors)
    {
        var payload = mapper.createObjectNode();
        payload.put("selectors", selectors);
        send(
            createEvent(
                tsMillis,
                "operation_started",
                login,
                containerType,
                containerId,
                operationId,
                payload));
    }

    @Override
    public void operationCancelled(
        long tsMillis,
        String login,
        ContainerType containerType,
        String containerId,
        String operationId)
    {
        send(
            createEvent(
                tsMillis,
                "operation_cancelled",
                login,
                containerType,
                containerId,
                operationId,
                null));
    }

    @Override
    public void operationTerminated(
        long tsMillis,
        ContainerType containerType,
        String containerId,
        String operationId,
        DeleteMetricsOperationStatus status,
        String statusMessage,
        int permanentlyDeletedMetricsCount)
    {
        var payload = mapper.createObjectNode();
        payload.put("status", status.name());
        payload.put("statusMessage", statusMessage);
        payload.put("permanentlyDeletedMetricsCount", permanentlyDeletedMetricsCount);
        send(
            createEvent(
                tsMillis,
                "operation_terminated",
                "",
                containerType,
                containerId,
                operationId,
                payload));
    }

    @Override
    public void close() throws Exception {
        session.close();
    }

    private ObjectNode createEvent(
        long tsMillis,
        String eventType,
        String login,
        ContainerType containerType,
        String containerId,
        String operationId,
        @Nullable ObjectNode payload)
    {
        var event = mapper.createObjectNode();
        event.put("ts", String.valueOf(millisecondsToSeconds(tsMillis)));
        event.put("eventType", eventType);
        event.put("login", login);
        event.put("containerType", containerType.name());
        event.put("containerId", containerId);
        event.put("operationId", operationId);
        event.set("payload", payload);
        return event;
    }

    private void send(ObjectNode event) {
        try {
            session.send(mapper.writeValueAsString(event));
        } catch (Throwable t) {
            logger.error("fail to send an event: {}", event, t);
        }
    }
}
