package ru.yandex.solomon.gateway.push;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.concurrent.ThreadSafe;

import org.springframework.stereotype.Component;

import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;


/**
 * @author alexlovkov
 **/
@Component
@LinkedOnRootPage("Push disabler")
@ThreadSafe
public class PushDisabler {

    private final Set<String> disabledFqdns;
    private final Set<String> disabledDcs;
    private final ClusterMapper clusterMapper;

    PushDisabler(ClusterMapper clusterMapper) {
        this.clusterMapper = clusterMapper;
        this.disabledFqdns = Collections.newSetFromMap(new ConcurrentHashMap<>());
        this.disabledDcs = Collections.newSetFromMap(new ConcurrentHashMap<>());
    }

    void disableFqdn(@ManagerMethodArgument(name = "fqdn") String fqdn) {
        disabledFqdns.add(fqdn);
    }

    void enableFqdn(@ManagerMethodArgument(name = "fqdn") String fqdn) {
        disabledFqdns.remove(fqdn);
    }

    void disableDc(@ManagerMethodArgument(name = "dc") String dc) {
        disabledDcs.add(dc);
    }

    void enableDc(@ManagerMethodArgument(name = "dc") String dc) {
        disabledDcs.remove(dc);
    }

    void clear() {
        disabledDcs.clear();
        disabledFqdns.clear();
    }

    List<String> filterFqdn(List<String> fqdns) {
        List<String> filteredFqdns = new ArrayList<>(fqdns.size());
        for (String fqdn : fqdns) {
            if (disabledFqdns.contains(fqdn)) {
                continue;
            }

            String dc = clusterMapper.byFqdnOrNull(fqdn);
            if (dc != null && disabledDcs.contains(dc)) {
                continue;
            }

            filteredFqdns.add(fqdn);
        }
        return filteredFqdns;
    }
}
