package ru.yandex.solomon.gateway.push;


import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monitoring.coremon.TPushedDataRequest;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.auth.exceptions.AuthorizationException;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.UnknownProjectException;
import ru.yandex.solomon.core.conf.UnknownShardException;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypes;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.coremon.client.CoremonClientContext;
import ru.yandex.solomon.coremon.client.ShardInfo;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.gateway.api.cloud.v2.MonitoringDataController;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.util.UnknownShardLocation;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.solomon.util.time.InstantUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;


/**
 * @author checat
 * @author Sergey Polovko
 */
@LinkedOnRootPage("Push Metric Processor")
@Component
@Import({ CoremonClientContext.class, PushDisabler.class })
@ParametersAreNonnullByDefault
public class PushMetricProcessor implements MetricSupplier {
    private static final Logger logger = LoggerFactory.getLogger(PushMetricProcessor.class);

    private final CoremonClient coremonClient;
    private final PushDisabler pushDisabler;
    private final PushMetrics metrics;
    private final SolomonConfHolder solomonConfHolder;
    private final FeatureFlagsHolder featureFlagsHolder;

    private volatile LastPushExceptionInfo lastPushException; // only for Manager UI

    @Autowired
    public PushMetricProcessor(
        CoremonClient coremonClient,
        PushDisabler pushDisabler,
        SolomonConfHolder solomonConfHolder,
        FeatureFlagsHolder featureFlagsHolder)
    {
        this.coremonClient = coremonClient;
        this.pushDisabler = pushDisabler;
        this.solomonConfHolder = solomonConfHolder;
        this.featureFlagsHolder = featureFlagsHolder;
        this.metrics = new PushMetrics();
    }

    public CompletableFuture<TDataProcessResponse> processPush(PushRequest request, boolean onlyNewFormatWrites) {
        ShardConfDetailed shardConf;
        try {
            shardConf = getShardConf(request.getShardKey(), request.getAccount());
        } catch (Throwable t) {
            return failedFuture(t);
        }

        int maxResponseSizeBytes;
        if (shardConf != null) {
            maxResponseSizeBytes = shardConf.getMaxResponseSizeBytes();
        } else {
            maxResponseSizeBytes = Shard.DEFAULT_RESPONSE_SIZE_QUOTA;
        }

        if (request.getContentSizeBytes() > maxResponseSizeBytes) {
            return completedFuture(TDataProcessResponse.newBuilder()
                    .setStatus(UrlStatusType.RESPONSE_TOO_LARGE)
                    .setErrorMessage("request size limit is " + maxResponseSizeBytes + " bytes, but request is " + request.getContentSizeBytes() + " bytes")
                    .build());
        }

        String service = request.getShardKey().getService();
        boolean isIam = request.getAccount().getAuthType() == AuthType.IAM;
        boolean isPublicService = MonitoringDataController.PUBLIC_SERVICES.contains(service);

        if (isIam && !isPublicService) {
            String project = request.getShardKey().getProject();
            if (!featureFlagsHolder.hasFlag(FeatureFlag.INTERNAL_CLOUD, project)) {
                SolomonConfWithContext conf = solomonConfHolder.getConfOrThrow();
                ServiceProvider serviceProvider = conf.getServiceProvider(service);
                if (serviceProvider == null) {
                    metrics.incInvalidServicePush(service, project);

                    return completedFuture(TDataProcessResponse.newBuilder()
                            .setStatus(UrlStatusType.AUTH_ERROR)
                            .setErrorMessage("unknown service provider: " + service)
                            .build());
                }
            }
        }

        CompletableFuture<ShardInfo> createShardFuture;
        if (shardConf != null) {
            ShardState state = shardConf.getRaw().getState();
            if (state == ShardState.INACTIVE || state == ShardState.READ_ONLY) {
                return CompletableFuture.completedFuture(TDataProcessResponse.newBuilder()
                        .setStatus(UrlStatusType.SHARD_IS_NOT_WRITABLE)
                        .setErrorMessage(String.format(
                                "shard %s in state %s, write is impossible",
                                request.getShardKey(), state))
                        .build());
            }

            List<String> hosts;
            try {
                hosts = pushDisabler.filterFqdn(coremonClient.shardHosts(shardConf.getNumId()));
            } catch (UnknownShardLocation e) {
                return completedFuture(TDataProcessResponse.newBuilder()
                        .setStatus(UrlStatusType.UNKNOWN_SHARD)
                        .setErrorMessage("shard " + request.getShardKey() + " was not found")
                        .build());
            }

            var fakeCreateResult = new ShardInfo(shardConf.getId(), shardConf.getNumId(), ImmutableList.copyOf(hosts));
            createShardFuture = completedFuture(fakeCreateResult);
        } else {
            createShardFuture = createShard(request.getShardKey(), request.getAccount());
        }

        var future = createShardFuture.thenCompose(shardInfo -> pushToAllHosts(shardInfo, request, onlyNewFormatWrites));
        return metrics.getAsyncMetrics().wrapFuture(future);
    }

    private CompletableFuture<ShardInfo> createShard(ShardKey shardKey, Account account) {
        try {
            IdValidator.ensureProjectIdValid(shardKey.getProject());
            IdValidator.ensureValid(shardKey.getCluster(), "cluster");
            IdValidator.ensureValid(shardKey.getService(), "service");
        } catch (Exception e) {
            return failedFuture(new BadRequestException("invalid shard key: " + shardKey + ", reason: " + e.getMessage()));
        }

        return coremonClient.createShard(shardKey.getProject(), shardKey.getCluster(), shardKey.getService(), account.getId())
            .whenComplete((r, t) -> {
                metrics.incShardsCreated(shardKey.getProject(), t);
                if (t != null) {
                    logger.error("cannot create shard " + shardKey + " on push, message: " + t.getMessage());
                } else {
                    logger.info("created shard " + r + " by push");
                }
            });
    }

    @Nullable
    private ShardConfDetailed getShardConf(ShardKey shardKey, Account account) {
        SolomonConfWithContext conf = solomonConfHolder.getConfOrThrow();
        ShardConfDetailed shardConf = conf.findShardOrNull(shardKey);
        if (shardConf != null) {
            if (account.isAnonymous()) {
                String projectId = shardKey.getProject();
                Project project = conf.getProject(projectId);

                if (project == null) {
                    throw new UnknownProjectException("cannot push to unknown project " + projectId);
                }

                boolean onlyAuthPush = project.isOnlyAuthPush();
                if (onlyAuthPush) {
                    String message = String.format(
                        "cannot push to project %s without authorization, use /api/v2/push endpoint instead",
                        projectId);
                    throw new AuthorizationException(message);
                }
            }
            return shardConf;
        }

        if (account.isAnonymous()) {
            // forbid push into unknown shard in case of anonymous access
            throw new UnknownShardException(shardKey);
        }

        // new shard must be created
        return null;
    }

    private void handleError(String shardId, ShardKey shardKey, MetricFormat format, Throwable x, AuthType authType) {
        Throwable cause = CompletableFutures.unwrapCompletionException(x);
        if (UrlStatusTypes.classifyError(cause) == UrlStatusType.UNKNOWN_ERROR) {
            logger.error("Unhandled exception while push into " + shardKey, cause);
        }
        lastPushException = new LastPushExceptionInfo(cause, System.currentTimeMillis(), shardKey);
        metrics.add(shardKey.getProject(), shardId, format, UrlStatusTypes.classifyError(cause), 0, authType);
    }

    private CompletableFuture<TDataProcessResponse> pushToAllHosts(
            ShardInfo shardInfo,
            PushRequest request,
            boolean onlyNewFormatWrites)
    {
        TPushedDataRequest pushRequest = TPushedDataRequest.newBuilder()
            .setNumId(shardInfo.getNumId())
            .setFormat(request.getFormat())
            .setContent(ByteStrings.fromByteBuf(request.takeContent()))
            .setOnlyNewFormatWrites(onlyNewFormatWrites)
            .setTimeMillis(InstantUtils.secondsToMillis(InstantUtils.currentTimeSeconds())) // round to seconds
            .build();

        var promise = new CompletableFuture<TDataProcessResponse>();
        var cnt = new AtomicInteger(shardInfo.getHosts().size());
        for (String host : shardInfo.getHosts()) {
            coremonClient.processPushedData(host, pushRequest)
                .whenComplete((response, throwable) -> {
                    String shardId = shardInfo.getShardId();
                    ShardKey shardKey = request.getShardKey();
                    MetricFormat format = request.getFormat();
                    AuthType authType = request.getAccount().getAuthType();

                    // complete promise on first OK or on last (with any status) response
                    final boolean isLast = cnt.decrementAndGet() == 0;
                    if (throwable == null) {
                        metrics.add(shardKey.getProject(), shardId, format, response.getStatus(), response.getSuccessMetricCount(), authType);
                        if (response.getStatus() == UrlStatusType.OK || isLast) {
                            promise.complete(response);
                        }
                    } else if (isLast) {
                        handleError(shardId, shardKey, format, throwable, authType);
                        promise.completeExceptionally(throwable);
                    }
                });
        }
        return promise;
    }

    @Override
    public int estimateCount() {
        return metrics.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        metrics.append(tsMillis, commonLabels, consumer);
    }

    /**
     * LAST PUSH EXCEPTION INFO
     */
    private static final class LastPushExceptionInfo {
        private final Throwable exception;
        private final long timeMillis;
        private final ShardKey shard;

        LastPushExceptionInfo(Throwable exception, long timeMillis, @Nullable ShardKey shard) {
            this.exception = exception;
            this.timeMillis = timeMillis;
            this.shard = shard;
        }

        @Override
        public String toString() {
            return "on shard: \'" + shard +
                "\', error: \'" + exception.getMessage() +
                "\', at " + InstantUtils.formatToMillis(timeMillis);
        }
    }
}
