package ru.yandex.solomon.gateway.push;

import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * Metrics to detect push status
 * <pre>
 *     push.started{}
 *     push.completedOk{}
 *     push.completedError{}
 *     push.inFlight{}
 *     push.responseTimeMillis{bin=*}
 *     push.shard.minSensorsProcessed{projectId=junk,shardId=junk_foo_bar,metricFormat=JSON}
 *     push.shard.maxSensorsProcessed{projectId=junk,shardId=junk_foo_bar,metricFormat=JSON}
 *     push.shard.request.count{projectId=junk,shardId=junk_foo_bar,metricFormat=JSON, pushStatus=OK}
 *     push.shard.request.count{projectId=junk,shardId=junk_foo_bar,authType=OAUTH}
 * </pre>
 *
 * @author Maksim Leonov (nohttp@)
 */
@Component
@ParametersAreNonnullByDefault
class PushMetrics implements MetricSupplier {
    // Prevent evict rare push for example once a day
    private static final long CACHE_TTL_MILLIS = TimeUnit.HOURS.toMillis(32L);

    private final MetricRegistry registry;

    private final AsyncMetrics asyncMetrics;

    private final Cache<ShardKey, ShardPushMetrics> statsByShard;
    private final Cache<String, ShardPushMetrics> statsByProjectId;
    private final Cache<Labels, InvalidPushes> invalidPushes;

    private final ShardPushMetrics totalStats;

    PushMetrics() {
        this.registry = new MetricRegistry();
        this.asyncMetrics = new AsyncMetrics(registry, "push");
        this.statsByShard = CacheBuilder.newBuilder()
                .expireAfterAccess(CACHE_TTL_MILLIS, TimeUnit.MILLISECONDS)
                .build();
        this.statsByProjectId = CacheBuilder.newBuilder()
                .expireAfterAccess(CACHE_TTL_MILLIS, TimeUnit.MILLISECONDS)
                .build();
        this.totalStats = new ShardPushMetrics("total", "total");
        this.invalidPushes = CacheBuilder.newBuilder()
                .expireAfterAccess(CACHE_TTL_MILLIS, TimeUnit.MILLISECONDS)
                .build();
    }

    void add(String projectId, String shardId, MetricFormat format, UrlStatusType status, int metricsProcessed, AuthType authType) {
        getStatsForShard(projectId, shardId).add(format, status, metricsProcessed, authType);
        getStatsForProject(projectId).add(format, status, metricsProcessed, authType);
        totalStats.add(format, status, metricsProcessed, authType);
    }

    void incShardsCreated(String projectId, @Nullable Throwable t) {
        getStatsForProject(projectId).incShardsCreated(t);
    }

    AsyncMetrics getAsyncMetrics() {
        return asyncMetrics;
    }

    void incInvalidServicePush(String service, String cloudId) {
        Labels labels = Labels.of("cloudId", cloudId, "serviceProvider", service);
        Labels totalLabels = Labels.of("cloudId", "total", "serviceProvider", service);
        incInvalidServicePush(labels);
        incInvalidServicePush(totalLabels);
    }

    private void incInvalidServicePush(Labels labels) {
        var metrics = invalidPushes.getIfPresent(labels);
        if (metrics == null) {
            metrics = new InvalidPushes(labels);
            invalidPushes.put(labels, metrics);
        }
        metrics.inc();
    }

    private ShardPushMetrics getStatsForShard(String projectId, String shardId) {
        var key = new ShardKey(projectId, shardId);
        var metrics = statsByShard.getIfPresent(key);
        if (metrics == null) {
            metrics = new ShardPushMetrics(projectId, shardId);
            statsByShard.put(key, metrics);
        }
        return metrics;
    }

    private ShardPushMetrics getStatsForProject(String projectId) {
        var metrics = statsByProjectId.getIfPresent(projectId);
        if (metrics == null) {
            metrics = new ShardPushMetrics(projectId, "total");
            statsByProjectId.put(projectId, metrics);
        }
        return metrics;
    }

    @Override
    public int estimateCount() {
        int total = registry.estimateCount();
        total += totalStats.estimateCount();
        total += statsByProjectId.size() * totalStats.estimateCount();
        total += statsByShard.size() * totalStats.estimateCount();
        total += invalidPushes.size() * 2;
        return total;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
        totalStats.append(tsMillis, commonLabels, consumer);
        for (var metrics : statsByProjectId.asMap().values()) {
            metrics.append(tsMillis, commonLabels, consumer);
        }
        for (var metrics : statsByShard.asMap().values()) {
            metrics.append(tsMillis, commonLabels, consumer);
        }
        for (var metrics : invalidPushes.asMap().values()) {
            metrics.append(tsMillis, commonLabels, consumer);
        }
    }

    private static record ShardKey(String projectId, String shardId) {
    }

    private static class InvalidPushes implements MetricSupplier {
        private final MetricRegistry registry;
        private final Counter counter;
        private final Rate rate;

        public InvalidPushes(Labels labels) {
            registry = new MetricRegistry();
            counter = registry.counter("push.invalid_service_requests.total", labels);
            rate = registry.rate("push.invalid_service_requests.rate", labels);
        }

        @Override
        public int estimateCount() {
            return 2;
        }

        public void inc() {
            counter.inc();
            rate.inc();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
        }
    }
}
