package ru.yandex.solomon.gateway.push;

import java.util.EnumMap;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Nullable;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;

/**
 * @author Oleg Baryshnikov
 */
class ShardPushMetrics implements MetricSupplier {

    private final MetricRegistry registry;

    private final ConcurrentHashMap<MetricFormat, PushShardFormatMetrics> metricsByFormat;
    private final PushShardFormatMetrics totalMetrics;
    private final Rate totalShardsCreatedOk;
    private final Rate totalShardsCreatedFail;
    private final EnumMap<AuthType, Rate> rpsPerAuth;

    ShardPushMetrics(String projectId, String shardId) {
        var labels = Labels.of("projectId", projectId, "shardId", shardId);
        this.registry = new MetricRegistry(labels);

        this.metricsByFormat = new ConcurrentHashMap<>();
        this.totalMetrics = new PushShardFormatMetrics(registry, "total");

        if ("total".equals(shardId)) {
            this.totalShardsCreatedOk = registry.rate("push.shards.created.ok");
            this.totalShardsCreatedFail = registry.rate("push.shards.created.fail");
        } else {
            // fake counters (not linked to the registry)
            this.totalShardsCreatedOk = this.totalShardsCreatedFail = new Rate();
        }

        this.rpsPerAuth = EnumMetrics.rates(AuthType.class, registry, "push.shard.request.count", "authType");
    }

    void incShardsCreated(@Nullable Throwable t) {
        if (t != null) {
            totalShardsCreatedFail.inc();
        } else {
            totalShardsCreatedOk.inc();
        }
    }

    void add(MetricFormat format, UrlStatusType status, int metricsProcessed, AuthType authType) {
        createPushShardFormatMetrics(format).add(status, metricsProcessed);
        totalMetrics.add(status, metricsProcessed);
        rpsPerAuth.get(authType).inc();
    }

    private PushShardFormatMetrics createPushShardFormatMetrics(MetricFormat format) {
        return metricsByFormat.computeIfAbsent(format, _format -> {
            String name = StringUtils.removeEnd(_format.name(), "_FORMAT");
            return new PushShardFormatMetrics(registry, name);
        });
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }
}
