package ru.yandex.solomon.gateway.stub.dao.ydb;

import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.transaction.TxControl;

import ru.yandex.solomon.gateway.stub.StubRequest;
import ru.yandex.solomon.gateway.stub.dao.StubRequestDao;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.idempotency.IdempotentOperationExistException;
import ru.yandex.solomon.idempotency.dao.ydb.YdbIdempotentOperationDao;

import static com.yandex.ydb.table.values.PrimitiveValue.string;
import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Nuradil Zhambyl
 */
@ParametersAreNonnullByDefault
public class YdbStubRequestDao implements StubRequestDao {
    private final SessionRetryContext retryCtx;
    private final SchemeClient schemeClient;
    private final YdbStubRequestQuery query;

    public YdbStubRequestDao(String root, String operationsRoot, TableClient tableClient, SchemeClient schemeClient) {
        this.schemeClient = schemeClient;
        this.query = new YdbStubRequestQuery(root, operationsRoot);
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();
    }

    @Override
    public CompletableFuture<Void> createSchemaForTests() {
        return schemeClient.makeDirectories(query.root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(unused -> createTable(query.table, YdbStubRequestTable::createTable));
    }

    @Override
    public CompletableFuture<Void> dropSchemaForTests() {
        Function<Session, CompletableFuture<Status>> doDrop = (session) -> session.dropTable(query.table);
        return retryCtx.supplyStatus(doDrop)
                .thenAccept(status -> status.expect("cannot drop " + query.table + " table"));
    }

    private CompletableFuture<Void> createTable(String tablePath, BiFunction<String, Session, CompletableFuture<Status>> fn) {
        return schemeClient.describePath(tablePath)
                .thenCompose(exist -> {
                    if (exist.isSuccess()) {
                        return completedFuture(com.yandex.ydb.core.Status.SUCCESS);
                    }

                    return retryCtx.supplyStatus(session -> fn.apply(tablePath, session));
                })
                .thenAccept(status -> status.expect("cannot create stub request table " + tablePath));
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return retryCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<StubRequest>> get(String id) {
        Params params = Params.of(
                YdbStubRequestTable.PARAM_ID, utf8(id)
        );
        return execute(query.selectOne, params)
                .thenApply(result -> {
                    var dataQueryResult = result.expect("unable select stub request by id");
                    var rs = dataQueryResult.getResultSet(0);
                    if (!rs.next()) {
                        return Optional.empty();
                    }

                    return Optional.of(request(rs));
                });
    }

    @Override
    public CompletableFuture<Boolean> insert(StubRequest request, IdempotentOperation op) {
        var params = Params.create()
                .put(YdbStubRequestTable.PARAM_ID, utf8(request.id()))
                .put(YdbStubRequestTable.PARAM_SERVICE_PROVIDER_ID, utf8(request.serviceProviderId()))
                .put(YdbStubRequestTable.PARAM_TYPE, utf8(request.type()))
                .put(YdbStubRequestTable.PARAM_STUB, string(request.stub().toByteString()))
                .put(YdbStubRequestTable.PARAM_EXECUTED_AT, timestamp(Instant.ofEpochMilli(request.executedAt())));
        YdbIdempotentOperationDao.fillExternalParams(params, op);
        try {
            return execute(query.insert, params)
                    .thenCompose(result -> handleInsertQueryResult(op, result, "stub request creation failed, no idempotent operation"));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Boolean> handleInsertQueryResult(IdempotentOperation op, Result<DataQueryResult> result, String failureMessage) {
        if (!result.isSuccess() && result.getCode() == StatusCode.PRECONDITION_FAILED && !op.isNoOperation()) {
            return ensureOperationDone(op, result, failureMessage);
        }
        result.expect("cannot execute query");
        return CompletableFuture.completedFuture(Boolean.TRUE);
    }

    private CompletableFuture<Boolean> handleDeleteQueryResult(IdempotentOperation op, Result<DataQueryResult> result, String failureMessage) {
        if (!result.isSuccess() && result.getCode() == StatusCode.PRECONDITION_FAILED && !op.isNoOperation()) {
            return ensureOperationDone(op, result, failureMessage);
        }
        var res = result.expect("cannot execute query");
        ResultSetReader resultSet = res.getResultSet(0);
        if (resultSet.next()) {
            return CompletableFuture.completedFuture(resultSet.getColumn(0).getBool());
        }
        return CompletableFuture.completedFuture(Boolean.FALSE);
    }

    private <T> CompletableFuture<T> ensureOperationDone(IdempotentOperation op, Result<DataQueryResult> parentResult, String failureMessage) {
        return execute(query.operationExists, YdbIdempotentOperationDao.existParams(op))
                .thenApply(result -> {
                    var res = result.expect("cannot execute query");
                    ResultSetReader resultSet = res.getResultSet(0);
                    if (resultSet.next() && resultSet.getColumn(0).getBool()) {
                        // handle if needed
                        throw new IdempotentOperationExistException();
                    }
                    // here is exception
                    parentResult.expect(failureMessage);
                    return null;
                });
    }

    @Override
    public CompletableFuture<Boolean> deleteOne(String id, IdempotentOperation op) {
        var params = Params.create()
                .put(YdbStubRequestTable.PARAM_ID, utf8(id));
        YdbIdempotentOperationDao.fillExternalParams(params, op);
        try {
            return execute(query.deleteOne, params)
                    .thenCompose(result -> handleDeleteQueryResult(op, result, "stub request deletion failed, no idempotent operation"));
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public static StubRequest request(ResultSetReader rs) {
        return new StubRequest(
                rs.getColumn(YdbStubRequestTable.ID).getUtf8(),
                rs.getColumn(YdbStubRequestTable.SERVICE_PROVIDER_ID).getUtf8(),
                rs.getColumn(YdbStubRequestTable.TYPE).getUtf8(),
                any(rs, YdbStubRequestTable.STUB),
                rs.getColumn(YdbStubRequestTable.EXECUTED_AT).getTimestamp().toEpochMilli()
        );
    }

    static Any any(ResultSetReader rs, String columnName) {
        try {
            return Any.parseFrom(rs.getColumn(columnName).getString());
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to parse column " + columnName + " at row " + rs.getRowCount());
        }
    }
}
