package ru.yandex.solomon.gateway.tasks;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Status;
import io.grpc.Status.Code;

import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.scheduler.proto.Task.State;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class RemoteTask implements SubTask<RemoteTaskProgress> {

    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
    private final AtomicReference<RemoteTaskProgress> progress;
    private final RemoteTaskClient remoteTaskClient;
    private final Predicate<RemoteTaskProgress> isComplete;

    private final AtomicInteger fetchCount = new AtomicInteger();
    private final PingActorRunner actor;

    private Runnable onProgressChange;

    public RemoteTask(
        String type,
        Executor executor,
        ScheduledExecutorService timer,
        RemoteTaskProgress progress,
        RemoteTaskClient remoteTaskClient,
        Predicate<RemoteTaskProgress> isComplete)
    {
        this.progress = new AtomicReference<>(progress);
        this.remoteTaskClient = remoteTaskClient;
        this.isComplete = isComplete;
        this.actor = PingActorRunner.newBuilder()
            .operation(type + "_remote_task_status")
            .executor(executor)
            .timer(timer)
            .pingInterval(Duration.ofSeconds(10))
            .onPing(this::actFetch)
            .build();
    }

    @Override
    public CompletableFuture<Void> start(boolean interrupt, Runnable onProgressChange) {
        try {
            return tryStart(interrupt, onProgressChange);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    @Override
    public boolean isIdle() {
        var last = progress();
        if (last.getRemoteTaskId().isEmpty()) {
            return false;
        }

        var remoteTask = last.getRemoteTask();
        if (remoteTask.getState() != State.SCHEDULED) {
            return false;
        }

        if (fetchCount.get() == 0) {
            return false;
        }

        var delay = System.currentTimeMillis() - remoteTask.getExecuteAt();
        return Math.abs(delay) > TimeUnit.SECONDS.toMillis(5);
    }

    @Override
    public RemoteTaskProgress progress() {
        return progress.get();
    }

    private CompletableFuture<Void> tryStart(boolean interrupt, Runnable onProgressChange) {
        if (progress().getComplete()) {
            return completedFuture(null);
        }

        this.onProgressChange = onProgressChange;

        if (interrupt) {
            return interruptRemoteTask()
                .thenCompose(exists -> {
                    if (exists) {
                        return watchTask();
                    }

                    completeNothingToInterrupt();
                    return completedFuture(null);
                });
        }

        return scheduleRemoteTask()
            .thenCompose(i -> watchTask());
    }

    private CompletableFuture<?> actFetch(int attempt) {
        if (fetchCount.get() > 0 && progress().getRemoteTask().getExecuteAt() > System.currentTimeMillis()) {
            return completedFuture(null);
        }

        return fetchTask()
                .whenComplete((complete, e) -> {
                    if (e != null) {
                        actor.close();
                        doneFuture.completeExceptionally(e);
                    } else if (complete) {
                        actor.close();
                        doneFuture.complete(null);
                    }
                });
    }

    private CompletableFuture<Void> watchTask() {
        return fetchTask()
            .thenCompose(remoteComplete -> remoteComplete ? completedFuture(null) : pollingRemoteTask())
            .whenComplete((ignore, e) -> complete(e == null));
    }

    private CompletableFuture<Boolean> interruptRemoteTask() {
        if (progress().getInterrupted()) {
            return completedFuture(Boolean.TRUE);
        }

        return remoteTaskClient.interrupt()
            .thenCompose(taskId -> {
                if (taskId.isEmpty()) {
                    return completedFuture(Boolean.FALSE);
                }

                progress.set(
                    progress().toBuilder()
                        .setInterrupted(true)
                        .setRemoteTaskId(taskId)
                        .build());
                progressChanged();

                return completedFuture(Boolean.TRUE);
            });
    }

    private CompletableFuture<Void> scheduleRemoteTask() {
        if (!progress().getRemoteTaskId().isEmpty()) {
            return completedFuture(null);
        }

        return remoteTaskClient.schedule()
                .thenAccept(taskId -> {
                    if (taskId.isEmpty()) {
                        throw Status.INVALID_ARGUMENT
                                .withDescription("task id empty")
                                .asRuntimeException();
                    }

                    progress.set(
                        progress().toBuilder()
                            .setRemoteTaskId(taskId)
                            .build());
                    progressChanged();
                });
    }

    private CompletableFuture<Boolean> fetchTask() {
        var last = progress();
        return fetchTask(last.getRemoteTaskId())
                .thenApply(task -> {
                    updateProgress(task);
                    return task.getState() == State.COMPLETED;
                })
                .exceptionally(e -> {
                    var status = Status.fromThrowable(e);
                    if (status.getCode() != Code.NOT_FOUND) {
                        throw new RuntimeException(e);
                    }

                    updateProgress(null);
                    if (progress().getRemoteTaskRemovedAt() == 0) {
                        throw new RuntimeException(e);
                    }

                    return Boolean.TRUE;
                });
    }

    private CompletableFuture<Task> fetchTask(String taskId) {
        var req = GetTaskRequest.newBuilder().setId(taskId).build();
        return remoteTaskClient.getTask(req);
    }

    private long timestampOrNow(long timestamp) {
        if (timestamp != 0) {
            return timestamp;
        }

        return System.currentTimeMillis();
    }

    private CompletableFuture<Void> pollingRemoteTask() {
        actor.forcePing();
        return doneFuture;
    }

    private void updateProgress(@Nullable Task task) {
        RemoteTaskProgress prev;
        RemoteTaskProgress.Builder update;
        do {
            prev = progress.get();
            update = prev.toBuilder();

            if (task != null) {
                update.setRemoteTask(task);
                if (task.getState() == State.COMPLETED) {
                    update.setRemoteTaskCompletedAt(timestampOrNow(prev.getRemoteTaskCompletedAt()));
                }
            } else if (prev.getRemoteTaskCompletedAt() > 0) {
                update.setRemoteTaskRemovedAt(timestampOrNow(prev.getRemoteTaskRemovedAt()));
            }
        } while (!progress.compareAndSet(prev, update.build()));
        fetchCount.incrementAndGet();
        progressChanged();
    }

    private void completeNothingToInterrupt() {
        progress.set(
            progress.get().toBuilder()
                .setComplete(true)
                .setInterrupted(true)
                .build());
        progressChanged();
    }

    private void complete(boolean success) {
        if (!success) {
            return;
        }

        RemoteTaskProgress prev;
        RemoteTaskProgress update;
        do {
            prev = progress.get();
            update = prev.toBuilder()
                    .setComplete(isComplete.test(prev))
                    .build();
        } while (!progress.compareAndSet(prev, update));
        progressChanged();
    }

    private void progressChanged() {
        var fn = onProgressChange;
        if (fn != null) {
            fn.run();
        }
    }

    @Override
    public void close() {
        remoteTaskClient.close();
        actor.close();
        doneFuture.completeExceptionally(Status.CANCELLED.asRuntimeException());
    }

    @ParametersAreNonnullByDefault
    public abstract static class RemoteTaskClient implements AutoCloseable {

        private final RetryContext retryCtx;

        protected RemoteTaskClient(RetryConfig retry) {
            this.retryCtx = new RetryContext(retry.withExceptionFilter(
                throwable -> Status.fromThrowable(throwable).getCode() != Code.NOT_FOUND));
        }

        protected CompletableFuture<String> interrupt() {
            throw new UnsupportedOperationException("this task does not support interruption");
        }

        protected abstract CompletableFuture<String> schedule();

        protected abstract CompletableFuture<Task> getTask(GetTaskRequest request);

        protected final <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
            return retryCtx.retry(supplier);
        }

        @Override
        public void close() {
            retryCtx.close();
        }
    }
}
