package ru.yandex.solomon.gateway.tasks;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.concurrent.CompletableFutures;

import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public final class ScatterGather<P> implements AutoCloseable {

    private final List<? extends SubTask<P>> subTasks;
    private final List<CompletableFuture<Void>> futures;

    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();

    public ScatterGather(List<? extends SubTask<P>> subTasks) {
        this.subTasks = subTasks;
        this.futures = new ArrayList<>(subTasks.size());
        for (int i = 0; i < subTasks.size(); i++) {
            this.futures.add(new CompletableFuture<>());
        }
    }

    public CompletableFuture<Void> start(boolean interrupt) {
        try {
            return tryStart(interrupt);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<Void> tryStart(boolean interrupt) {
        for (int index = 0; index < subTasks.size(); index++) {
            var future = subTasks.get(index).start(interrupt, this::onProgressChange);
            CompletableFutures.whenComplete(future, futures.get(index));
            future.whenComplete((ignore, e) -> onProgressChange());
        }

        return doneFuture;
    }

    private void onProgressChange() {
        int running = subTasks.size();
        for (int index = 0; index < subTasks.size(); index++) {
            if (futures.get(index).isDone() || subTasks.get(index).isIdle()) {
                running--;
            }
        }

        if (running > 0) {
            return;
        }

        try {
            for (var future : futures) {
                if (future.isCompletedExceptionally()) {
                    CompletableFutures.whenComplete(future, doneFuture);
                    return;
                }
            }

            doneFuture.complete(null);
        } finally {
            close();
        }
    }

    public List<P> progress() {
        var result = new ArrayList<P>(subTasks.size());
        for (var subTask : subTasks) {
            result.add(subTask.progress());
        }
        return result;
    }

    @Override
    public void close() {
        for (var subTask : subTasks) {
            subTask.close();
        }
    }
}
