package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.protobuf.Any;
import com.google.protobuf.TextFormat;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.gateway.api.task.DeleteMetricsProgress;
import ru.yandex.gateway.api.task.DeleteMetricsProgress.PhaseOnReplicaProgress;
import ru.yandex.gateway.api.task.DeleteMetricsResult;
import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest.Phase;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationMetrics;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationStatus;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationTracker;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.grpc.Proto;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.lang.System.currentTimeMillis;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.TimeUnit.DAYS;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.concurrent.TimeUnit.SECONDS;
import static java.util.function.Predicate.not;
import static ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager.unpackData;
import static ru.yandex.solomon.util.time.DurationUtils.backoff;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class DeleteMetricsTask implements AutoCloseable {
    private static final long BACKOFF_DELAY_MILLIS = MINUTES.toMillis(10);
    private static final long BACKOFF_MAX_DELAY_MILLIS = DAYS.toMillis(1);

    private static final Logger logger = LoggerFactory.getLogger(DeleteMetricsTask.class);

    private final ExecutionContext context;
    private final DeleteMetricsParams params;
    private final AtomicReference<DeleteMetricsProgress> progress;
    private final AtomicReference<LongRunningOperation> operation;

    private final SolomonConfHolder confHolder;
    private final DeleteMetricsOperationManager manager;
    private final DeleteMetricsOperationMetrics metrics;
    private final DeleteMetricsOperationTracker tracker;

    private final boolean rollbackRequested;
    private final PrepareOperation prepareOperation;
    private final List<PhaseOnReplica> checkOnReplicas;
    private final List<PhaseOnReplica> moveOnReplicas;
    private final List<PhaseOnReplica> rollbackOnReplicas;
    private final AtomicBoolean pointOfNoReturn;
    private final List<PhaseOnReplica> terminateOnReplicas;

    private final PingActorRunner actor;

    DeleteMetricsTask(
        RetryConfig retryConfig,
        SolomonConfHolder confHolder,
        CoremonClient coremonClient,
        DeleteMetricsOperationManager manager,
        DeleteMetricsOperationMetrics metrics,
        DeleteMetricsOperationTracker tracker,
        Executor executor,
        ScheduledExecutorService timer,
        ExecutionContext context)
    {
        this.context = context;
        this.params = DeleteMetricsTaskProto.params(context.task().params());
        this.progress = new AtomicReference<>(DeleteMetricsTaskProto.progress(context.task().progress()));
        this.operation = new AtomicReference<>();

        this.confHolder = confHolder;
        this.manager = manager;
        this.metrics = metrics;
        this.tracker = tracker;

        retryConfig = retryConfig
            .withStats((timeSpentMillis, cause) -> logger.warn("{} error", logPrefix(), cause));

        var operationId = context.task().id();
        this.rollbackRequested = progress.get().hasRollbackRequested();
        this.prepareOperation = new PrepareOperation(
            retryConfig,
            manager,
            operationId,
            params);
        this.checkOnReplicas = newPhaseOnReplicas(
            Phase.CHECK,
            retryConfig,
            coremonClient,
            executor,
            timer,
            operationId,
            params,
            progress.get().getCheckOnReplicasList());
        this.moveOnReplicas = newPhaseOnReplicas(
            Phase.MOVE,
            retryConfig,
            coremonClient,
            executor,
            timer,
            operationId,
            params,
            progress.get().getMoveOnReplicasList());
        this.rollbackOnReplicas = newPhaseOnReplicas(
            Phase.ROLLBACK,
            retryConfig,
            coremonClient,
            executor,
            timer,
            operationId,
            params,
            progress.get().getRollbackOnReplicasList());
        this.pointOfNoReturn = new AtomicBoolean(progress.get().getPointOfNoReturn());
        this.terminateOnReplicas = newPhaseOnReplicas(
            Phase.TERMINATE,
            retryConfig,
            coremonClient,
            executor,
            timer,
            operationId,
            params,
            progress.get().getTerminateOnReplicasList());
        this.actor = PingActorRunner.newBuilder()
            .pingInterval(Duration.ofSeconds(15))
            .operation(DeleteMetricsTaskHandler.TYPE + "_save_progress")
            .executor(executor)
            .timer(timer)
            .onPing(this::saveProgress)
            .build();
    }

    public CompletableFuture<Void> start() {
        actor.schedule();

        var root = new CompletableFuture<>();
        var future = root
            .thenCompose(ignore -> onStart())
            .handle((ignore, e) -> e != null ? Status.fromThrowable(e) : Status.OK)
            .thenCompose(status -> {
                updateProgress(status);
                if (context.isDone()) {
                    return completedFuture(null);
                }

                if (status.isOk()) {
                    return complete();
                }

                return reschedule();
            })
            .whenComplete((ignore, e) -> {
                if (e != null && logger.isWarnEnabled()) {
                    logger.warn("{} failed on complete", logPrefix(), e);
                }

                close();
            });

        root.complete(null);
        return future;
    }

    public DeleteMetricsProgress progress() {
        updateProgress();
        return progress.get();
    }

    @Override
    public void close() {
        if (logger.isDebugEnabled()) {
            logger.debug("{} close", logPrefix());
        }

        if (!context.isDone()) {
            context.cancel();
        }

        prepareOperation.close();
        checkOnReplicas.forEach(PhaseOnReplica::close);
        moveOnReplicas.forEach(PhaseOnReplica::close);
        rollbackOnReplicas.forEach(PhaseOnReplica::close);
        terminateOnReplicas.forEach(PhaseOnReplica::close);
        actor.close();
    }

    private CompletionStage<Void> onStart() {
        return prepareOperation();
    }

    private CompletableFuture<Void> prepareOperation() {
        return prepareOperation.prepare()
            .thenCompose(this::onCompletePrepareOperation);
    }

    private CompletableFuture<Void> onCompletePrepareOperation(LongRunningOperation operation) {
        this.operation.set(operation);
        return rollbackRequested
            ? interruptMoveOnReplicas()
            : checkOnReplicas();
    }

    private CompletableFuture<Void> checkOnReplicas() {
        return phaseOnReplicas(checkOnReplicas, 0, false, this::onCompleteCheckOnReplicas);
    }

    private CompletableFuture<Void> onCompleteCheckOnReplicas() {
        return moveOnReplicas();
    }

    private CompletableFuture<Void> moveOnReplicas() {
        return phaseOnReplicas(moveOnReplicas, 0, false, this::onCompleteMoveOnReplicas);
    }

    private CompletableFuture<Void> onCompleteMoveOnReplicas() {
        var terminateAt = effectivePermanentDeletionAt();
        var now = currentTimeMillis();
        if (terminateAt <= now) {
            return terminateOnReplicas();
        }

        return reschedule(terminateAt)
            .thenRun(() -> reportMoveCompleteOnReplicas(now));
    }

    private CompletableFuture<Void> terminateOnReplicas() {
        touchMetrics();
        pointOfNoReturn.set(true);
        return actor.forcePing()
            .thenCompose(i -> phaseOnReplicas(terminateOnReplicas, 0, false, this::onCompleteTerminateOnReplicas));
    }

    private CompletableFuture<Void> onCompleteTerminateOnReplicas() {
        return completedFuture(null);
    }

    private CompletableFuture<Void> interruptMoveOnReplicas() {
        return phaseOnReplicas(Lists.reverse(moveOnReplicas), 0, true, this::onCompleteInterruptMoveOnReplicas);
    }

    private CompletableFuture<Void> onCompleteInterruptMoveOnReplicas() {
        return interruptCheckOnReplicas();
    }

    private CompletableFuture<Void> interruptCheckOnReplicas() {
        return phaseOnReplicas(Lists.reverse(checkOnReplicas), 0, true, this::onCompleteInterruptCheckOnReplicas);
    }

    private CompletableFuture<Void> onCompleteInterruptCheckOnReplicas() {
        var terminateAt = effectivePermanentDeletionAt();
        if (terminateAt <= currentTimeMillis()) {
            return interruptRollbackOnReplicas();
        }

        return rollbackOnReplicas();
    }

    private CompletableFuture<Void> rollbackOnReplicas() {
        return phaseOnReplicas(rollbackOnReplicas, 0, false, this::onCompleteRollbackOnReplicas);
    }

    private CompletableFuture<Void> onCompleteRollbackOnReplicas() {
        // re-check in case rollback is still stuck on some replicas
        var executeAt = rollbackOnReplicas.stream()
            .flatMap(replica -> replica.progress().getOnShardsList().stream())
            .filter(not(RemoteTaskProgress::getComplete))
            .mapToLong(value -> value.getRemoteTask().getExecuteAt())
            .min();

        if (executeAt.isPresent()) {
            var rescheduleAt = Math.max(executeAt.getAsLong(), currentTimeMillis()) + randomize(SECONDS.toMillis(10));
            return reschedule(rescheduleAt);
        }

        return terminateOnReplicas();
    }

    private CompletableFuture<Void> interruptRollbackOnReplicas() {
        return phaseOnReplicas(rollbackOnReplicas, 0, true, this::onCompleteInterruptRollbackOnReplicas);
    }

    private CompletableFuture<Void> onCompleteInterruptRollbackOnReplicas() {
        return terminateOnReplicas();
    }

    private CompletableFuture<Void> phaseOnReplicas(
        List<PhaseOnReplica> phaseOnReplicas,
        int replicaIdx,
        boolean interrupt,
        Supplier<CompletableFuture<Void>> onComplete)
    {
        if (replicaIdx >= phaseOnReplicas.size()) {
            return onComplete.get();
        }

        var replica = phaseOnReplicas.get(replicaIdx);
        var nextIdx = replicaIdx + 1;

        return replica.start(interrupt)
            .thenCompose(i -> actor.forcePing())
            .thenCompose(i -> onCompletePhaseOnReplica(
                replica,
                interrupt,
                () -> phaseOnReplicas(phaseOnReplicas, nextIdx, interrupt, onComplete)));
    }

    private CompletableFuture<Void> onCompletePhaseOnReplica(
        PhaseOnReplica phaseOnReplica,
        boolean interrupt,
        Supplier<CompletableFuture<Void>> onSuccess)
    {
        var executeAt = phaseOnReplica.progress().getOnShardsList().stream()
            .filter(not(RemoteTaskProgress::getComplete))
            .mapToLong(value -> value.getRemoteTask().getExecuteAt())
            .min();

        if (executeAt.isPresent()) {
            if (rollbackRequested) {
                var isRollback = phaseOnReplica.progress().getOnShardsList().stream()
                    .anyMatch(shard -> "delete_metrics_rollback".equals(shard.getRemoteTask().getType()));

                if (isRollback) {
                    var stuck = phaseOnReplica.progress().getOnShardsList().stream()
                        .filter(not(RemoteTaskProgress::getComplete))
                        .allMatch(shard -> {
                            var remoteTask = shard.getRemoteTask();
                            var remoteProgress = DeleteMetricsTaskProto.remoteRollbackProgress(remoteTask.getProgress());

                            return remoteProgress.getStatus().getCode() != Status.Code.OK.value();
                        });

                    if (stuck) {
                        // we need to run rollback task on next replicas in case it was stuck on this one
                        // otherwise, other replicas will have no chance to rollback before termination at all
                        return onSuccess.get();
                    }
                }
            }

            var rescheduleAt = Math.max(executeAt.getAsLong(), currentTimeMillis()) + randomize(SECONDS.toMillis(10));
            return reschedule(rescheduleAt);
        }

        if (interrupt) {
            return onSuccess.get();
        }

        // due to current ScatterGather implementation we will wait for completion of the all shards
        // even if one of them has been already failed,
        // which is totally ok while deletion is restricted to a single shard,
        // but may become cumbersome for the CHECK phase in case of multi-shard deletions in future
        var anyError = phaseOnReplica.progress().getOnShardsList().stream()
            .filter(RemoteTaskProgress::getComplete)
            .map(shard -> Proto.fromProto(shard.getRemoteTask().getStatus()))
            .filter(status -> !status.isOk())
            .findAny();

        if (anyError.isPresent()) {
            return fail(anyError.get());
        }

        return onSuccess.get();
    }

    private CompletableFuture<Void> fail(Status status) {
        pointOfNoReturn.set(true);

        var now = currentTimeMillis();
        var progress = progress();
        var statusMessage = Strings.nullToEmpty(status.getDescription());

        return context.progress(progress)
            .thenCompose(i -> {
                var prev = this.operation.get();
                var data = unpackData(prev).toBuilder()
                    .setProgressPercentage(100)
                    .setStatusMessage(statusMessage)
                    .build();
                var update = prev.toBuilder()
                    .setUpdatedAt(now)
                    .setStatus(DeleteMetricsOperationStatus.METRICS_HAVE_RECENT_WRITES.value)
                    .setData(Any.pack(data))
                    .build();

                return manager.forceUpdateOperation(update)
                    .thenAccept(this::updateOperation);
            })
            .thenCompose(
                i -> context.fail(status.asRuntimeException())
                    .thenRun(() -> {
                        logger.info(
                            "{} fail {}, latest progress ({})",
                            logPrefix(),
                            status,
                            TextFormat.shortDebugString(this.progress.get()));

                        reportOperationTerminated(
                            now,
                            DeleteMetricsOperationStatus.METRICS_HAVE_RECENT_WRITES,
                            statusMessage,
                            0);
                    }));
    }

    private CompletableFuture<Void> complete() {
        var now = currentTimeMillis();
        var progress = progress();
        var result = DeleteMetricsTaskProto.mergeToResult(progress);

        var deletedMetricsCount = result.getResultsList().stream()
            .mapToInt(DeleteMetricsResult.ReplicaResult::getDeletedMetrics)
            .max()
            .orElse(0);

        DeleteMetricsOperationStatus status;
        if (rollbackRequested) {
            if (deletedMetricsCount == 0) {
                status = DeleteMetricsOperationStatus.CANCELLED;
            } else {
                status = DeleteMetricsOperationStatus.COMPLETED;
            }
        } else {
            status = DeleteMetricsOperationStatus.COMPLETED;
        }

        return context.progress(progress)
            .thenCompose(i -> {
                var prev = this.operation.get();
                var data = unpackData(prev).toBuilder()
                    .setProgressPercentage(100)
                    .setPermanentlyDeletedMetricsCount(deletedMetricsCount)
                    .build();
                var update = prev.toBuilder()
                    .setUpdatedAt(now)
                    .setStatus(status.value)
                    .setData(Any.pack(data))
                    .build();

                return manager.forceUpdateOperation(update)
                    .thenAccept(this::updateOperation);
            })
            .thenCompose(i -> {
                actor.close();
                return context.complete(result)
                    .thenRun(() -> {
                        logger.info(
                            "{} completed ({})",
                            logPrefix(),
                            TextFormat.shortDebugString(result));

                        reportOperationTerminated(now, status, "", deletedMetricsCount);
                    });
            });
    }

    private CompletableFuture<Void> reschedule(long executeAt) {
        var progress = progress();
        return saveOperation(progress)
            .thenCompose(i -> {
                actor.close();
                return context.reschedule(executeAt, progress)
                    .thenRun(() -> {
                        if (logger.isDebugEnabled()) {
                            logger.debug(
                                "{} rescheduled on {}, latest progress ({})",
                                logPrefix(),
                                Instant.ofEpochMilli(executeAt),
                                TextFormat.shortDebugString(progress));
                        }
                    });
            });
    }

    private CompletableFuture<Void> reschedule() {
        var progress = progress();
        long delayMillis = randomize(backoff(BACKOFF_DELAY_MILLIS, BACKOFF_MAX_DELAY_MILLIS, progress.getAttempt()));
        long executeAt = currentTimeMillis() + delayMillis;
        return reschedule(executeAt);
    }

    private boolean updateOperation(LongRunningOperation update) {
        LongRunningOperation prev;
        do {
            prev = operation.get();

            if (update.version() <= prev.version()) {
                return false;
            }
        } while (!operation.compareAndSet(prev, update));

        if (logger.isDebugEnabled()) {
            logger.debug("{} operation ({})", logPrefix(), update);
        }

        return true;
    }

    private boolean updateProgress() {
        return updateProgress(null);
    }

    private boolean updateProgress(@Nullable Status status) {
        DeleteMetricsProgress prev;
        DeleteMetricsProgress update;
        do {
            prev = progress.get();
            var updateBuilder = prev.toBuilder()
                .clearCheckOnReplicas()
                .addAllCheckOnReplicas(Lists.transform(checkOnReplicas, PhaseOnReplica::progress))
                .clearMoveOnReplicas()
                .addAllMoveOnReplicas(Lists.transform(moveOnReplicas, PhaseOnReplica::progress))
                .clearRollbackOnReplicas()
                .addAllRollbackOnReplicas(Lists.transform(rollbackOnReplicas, PhaseOnReplica::progress))
                .setPointOfNoReturn(pointOfNoReturn.get())
                .clearTerminateOnReplicas()
                .addAllTerminateOnReplicas(Lists.transform(terminateOnReplicas, PhaseOnReplica::progress));

            if (status != null) {
                if (status.isOk()) {
                    updateBuilder.clearAttempt();
                    updateBuilder.clearStatus();
                } else {
                    updateBuilder.setAttempt(prev.getAttempt() + 1)
                        .setStatus(Proto.toProto(status));
                }
            }

            update = updateBuilder.build();
            if (prev.equals(update)) {
                return false;
            }
        } while (!progress.compareAndSet(prev, update));

        if (logger.isDebugEnabled()) {
            logger.debug("{} progress ({})", logPrefix(), TextFormat.shortDebugString(update));
        }

        return true;
    }

    private CompletableFuture<?> saveProgress(int attempt) {
        if (context.isDone()) {
            close();
            return completedFuture(null);
        }

        if (!updateProgress()) {
            return completedFuture(null);
        }

        var progress = this.progress.get();
        return context.progress(progress)
            .thenCompose(i -> afterSaveProgress(progress))
            .whenComplete((ignore, e) -> {
                if (context.isDone()) {
                    close();
                }
            });
    }

    private CompletableFuture<Void> afterSaveProgress(DeleteMetricsProgress progress) {
        return saveOperation(progress);
    }

    private CompletableFuture<Void> saveOperation(DeleteMetricsProgress progress) {
        var prev = this.operation.get();
        if (DeleteMetricsOperationStatus.fromValue(prev.status()).isTerminal()) {
            return completedFuture(null);
        }

        var status = resolveDeleteMetricsOperationStatus(progress);

        var progressEstimation = rollbackRequested
            ? ProgressEstimation.onRollback(params, progress)
            : ProgressEstimation.onMove(params, progress);

        var data = unpackData(prev).toBuilder()
            .setPermanentDeletionAt(effectivePermanentDeletionAt())
            .setProgressPercentage(progressEstimation.progressPercentage())
            .setEstimatedMetricsCount(progressEstimation.estimatedMetrics())
            .setCancelledBy(progress.getRollbackRequested().getRequestedBy())
            .setCancelledAt(progress.getRollbackRequested().getRequestedAt())
            .setStatusMessage(renderStatusMessage(progressEstimation))
            .build();
        var update = prev.toBuilder()
            .setUpdatedAt(currentTimeMillis())
            .setStatus(status.value)
            .setData(Any.pack(data))
            .build();

        return manager.updateOperation(update)
            .thenAccept(updated -> updated.ifPresent(this::updateOperation));
    }

    private DeleteMetricsOperationStatus resolveDeleteMetricsOperationStatus(DeleteMetricsProgress progress) {
        var terminating = progress.getTerminateOnReplicasList().stream()
            .anyMatch(
                replica -> replica.getOnShardsList().stream()
                    .anyMatch(RemoteTaskProgress::hasRemoteTask));
        if (terminating) {
            return DeleteMetricsOperationStatus.DELETING_PERMANENTLY;
        }

        if (rollbackRequested) {
            return DeleteMetricsOperationStatus.CANCELLING;
        }

        var moved = progress.getMoveOnReplicasList().stream()
            .allMatch(
                replica -> replica.getOnShardsList().stream()
                    .allMatch(RemoteTaskProgress::getComplete));
        if (moved) {
            return DeleteMetricsOperationStatus.WAITING_FOR_PERMANENT_DELETION;
        }

        return DeleteMetricsOperationStatus.DELETING;
    }

    private String renderStatusMessage(ProgressEstimation progressEstimation) {
        if (progressEstimation.progressStatuses().isEmpty()) {
            return "";
        }

        var statusMessage = new StringBuilder(256);

        var conf = confHolder.getConfOrThrow();
        for (var s : progressEstimation.progressStatuses()) {
            var shardConf = conf.getShardByNumIdOrNull(s.numId());
            var shardDesc = shardConf != null
                ? shardConf.getId()
                : Integer.toUnsignedString(s.numId());

            statusMessage
                .append("Cancellation stuck on shard ")
                .append(shardDesc)
                .append(" [").append(s.clusterId()).append("]")
                .append(": ")
                .append(s.status())
                .append("\n");
        }

        return statusMessage.toString();
    }

    private long effectivePermanentDeletionAt() {
        return rollbackRequested
            ? params.getPermanentDeletionAt() + DAYS.toMillis(7)
            : params.getPermanentDeletionAt();
    }

    private void touchMetrics() {
        metrics.touch(params.getProjectId());
        for (int i = 0; i < params.getNumIdsCount(); i++) {
            var conf = confHolder.getConfOrThrow();
            var shard = conf.getShardByNumIdOrNull(params.getNumIds(i));
            if (shard != null) {
                metrics.touch(params.getProjectId(), shard.getId());
            }
        }
    }

    private void reportMoveCompleteOnReplicas(long now) {
        metrics.moveCompleteOnReplicas(
            params.getProjectId(),
            millisecondsToSeconds(now - params.getCreatedAt()));
    }

    private void reportOperationTerminated(
        long now,
        DeleteMetricsOperationStatus status,
        String statusMessage,
        int permanentlyDeletedMetricsCount)
    {
        metrics.operationTerminated(params.getProjectId(), status);
        for (int i = 0; i < params.getNumIdsCount(); i++) {
            var conf = confHolder.getConfOrThrow();
            var shard = conf.getShardByNumIdOrNull(params.getNumIds(i));
            if (shard != null) {
                metrics.operationTerminated(params.getProjectId(), shard.getId(), status);
            }
        }
        tracker.operationTerminated(
            now,
            ContainerType.PROJECT,
            params.getProjectId(),
            context.task().id(),
            status,
            statusMessage,
            permanentlyDeletedMetricsCount);
    }

    private String logPrefix() {
        return "delete_metrics(task_id: \"" + context.task().id() + "\" " + TextFormat.shortDebugString(params) + ")";
    }

    private static List<PhaseOnReplica> newPhaseOnReplicas(
        Phase phase,
        RetryConfig retry,
        CoremonClient coremonClient,
        Executor executor,
        ScheduledExecutorService timer,
        String operationId,
        DeleteMetricsParams params,
        List<PhaseOnReplicaProgress> progressOnReplicas)
    {
        var clusterIds = coremonClient.clusterIds();
        var unknownClusterId = new TreeSet<>(clusterIds);

        var progressInProcessingOrder = new ArrayList<PhaseOnReplicaProgress>(clusterIds.size());
        for (var progress : progressOnReplicas) {
            unknownClusterId.remove(progress.getClusterId());
            progressInProcessingOrder.add(progress);
        }
        for (var clusterId : unknownClusterId) {
            var progress = PhaseOnReplicaProgress.newBuilder()
                .setClusterId(clusterId)
                .build();
            progressInProcessingOrder.add(progress);
        }

        var replicas = new ArrayList<PhaseOnReplica>(progressInProcessingOrder.size());
        for (var progress : progressInProcessingOrder) {
            var replica = new PhaseOnReplica(
                phase,
                retry,
                coremonClient,
                executor,
                timer,
                operationId,
                params,
                progress);
            replicas.add(replica);
        }
        return replicas;
    }
}
