package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.Any;
import com.google.protobuf.Descriptors.Descriptor;
import io.grpc.Status;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.gateway.api.task.DeleteMetricsProgress;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.gateway.api.utils.IdGenerator;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationMetrics;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationTracker;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.PermitLimiter;
import ru.yandex.solomon.scheduler.ProgressOperator;
import ru.yandex.solomon.scheduler.ProgressOperator.Fail;
import ru.yandex.solomon.scheduler.ProgressOperator.Ok;
import ru.yandex.solomon.scheduler.ProgressOperator.Stop;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class DeleteMetricsTaskHandler implements TaskHandler {

    static final String TYPE = "delete_metrics";

    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
        .withNumRetries(10)
        .withDelay(TimeUnit.SECONDS.toMillis(1))
        .withMaxDelay(TimeUnit.MINUTES.toMillis(1));

    private final PermitLimiter permitLimiter = new PermitLimiter(10);

    private final SolomonConfHolder confHolder;
    private final CoremonClient coremon;
    private final DeleteMetricsOperationManager manager;
    private final DeleteMetricsOperationMetrics metrics;
    private final DeleteMetricsOperationTracker tracker;
    private final Executor executor;
    private final ScheduledExecutorService timer;

    // for manager ui
    private final ConcurrentMap<String, DeleteMetricsTask> running = new ConcurrentHashMap<>();

    public DeleteMetricsTaskHandler(
        SolomonConfHolder confHolder,
        CoremonClient coremon,
        DeleteMetricsOperationManager manager,
        DeleteMetricsOperationMetrics metrics,
        DeleteMetricsOperationTracker tracker,
        Executor executor,
        ScheduledExecutorService timer)
    {
        this.confHolder = confHolder;
        this.coremon = coremon;
        this.manager = manager;
        this.metrics = metrics;
        this.tracker = tracker;
        this.executor = executor;
        this.timer = timer;
    }

    @Override
    public String type() {
        return TYPE;
    }

    @Override
    @Nullable
    public Permit acquire(String id, Any params) {
        var conf = confHolder.getConf();
        if (conf == null) {
            return null;
        }

        if (coremon.clusterIds().isEmpty()) {
            return null;
        }

        return permitLimiter.acquire();
    }

    @Override
    public void execute(ExecutionContext context) {
        var taskId = context.task().id();
        var task = new DeleteMetricsTask(
            RETRY_CONFIG,
            confHolder,
            coremon,
            manager,
            metrics,
            tracker,
            executor,
            timer,
            context);
        running.put(taskId, task);
        try {
            task.start().whenComplete((ignore, e) -> {
                running.remove(taskId, task);
                task.close();
            });
        } catch (Throwable e) {
            running.remove(taskId, task);
            throw new RuntimeException(e);
        }
    }

    @SuppressWarnings("UnnecessaryFullyQualifiedName")
    @Override
    public List<Descriptor> descriptors() {
        return List.of(
            // gateway
            ru.yandex.gateway.api.task.DeleteMetricsParams.getDescriptor(),
            ru.yandex.gateway.api.task.DeleteMetricsProgress.getDescriptor(),
            ru.yandex.gateway.api.task.DeleteMetricsResult.getDescriptor(),
            // coremon
            ru.yandex.coremon.api.task.DeleteMetricsParams.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsCheckProgress.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsCheckResult.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsMoveProgress.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsMoveResult.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsRollbackProgress.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsRollbackResult.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsTerminateProgress.getDescriptor(),
            ru.yandex.coremon.api.task.DeleteMetricsTerminateResult.getDescriptor()
        );
    }

    public static Task task(DeleteMetricsParams params) {
        if (params.getContainerCase() != DeleteMetricsParams.ContainerCase.PROJECT_ID) {
            throw new UnsupportedOperationException("Not implemented container type: " + params.getContainerCase());
        }

        return Task.newBuilder()
            .setId(IdGenerator.generateInternalId())
            .setType(TYPE)
            .setExecuteAt(System.currentTimeMillis())
            .setParams(Any.pack(params))
            .build();
    }

    public static ProgressOperator newRollbackOperator(
        SolomonConfWithContext conf,
        String requestedBy,
        long now)
    {
        return progress -> {
            var prev = DeleteMetricsTaskProto.progress(progress);
            if (prev.hasRollbackRequested()) {
                return new Stop();
            }

            if (prev.getPointOfNoReturn()) {
                return new Fail(
                    Status.FAILED_PRECONDITION
                        .withDescription("too late")
                        .asRuntimeException());
            }

            var description = validateNoTtlInterference(Instant.ofEpochMilli(now), conf, prev);
            if (description != null) {
                return new Fail(
                    Status.FAILED_PRECONDITION
                        .withDescription(description)
                        .asRuntimeException());
            }

            var update = prev.toBuilder()
                .setRollbackRequested(
                    DeleteMetricsProgress.RollbackRequested.newBuilder()
                        .setRequestedBy(requestedBy)
                        .setRequestedAt(now))
                .build();

            return new Ok(Any.pack(update));
        };
    }

    /**
     * triggers only if check phase fully completed
     */
    @VisibleForTesting
    @Nullable
    static String validateNoTtlInterference(
        Instant now,
        SolomonConfWithContext conf,
        DeleteMetricsProgress progress)
    {
        @Nullable StringBuilder description = null;
        @Nullable IntSet processedNumIds = null;

        for (var replica : progress.getCheckOnReplicasList()) {
            for (var shard : replica.getOnShardsList()) {
                if (!shard.getComplete()) {
                    return null;
                }

                var remoteTask = shard.getRemoteTask();
                var checkParams = DeleteMetricsTaskProto.remoteParams(remoteTask.getParams());
                var checkResult = DeleteMetricsTaskProto.remoteCheckResult(remoteTask.getResult());

                var numId = checkParams.getNumId();
                var shardConf = conf.getShardByNumIdOrNull(numId);
                if (shardConf != null) {
                    var metricsTtlDays = shardConf.getConfOrThrow().getMetricsTtlDays();
                    if (metricsTtlDays <= 0) {
                        continue;
                    }

                    var expireTime = now.minus(metricsTtlDays, ChronoUnit.DAYS);

                    var minLastPoint = Instant.ofEpochSecond(checkResult.getMinLastPointSeconds());

                    var timeLeft = Duration.between(expireTime, minLastPoint);
                    if (timeLeft.compareTo(Duration.ofDays(1)) < 0) {
                        if (description == null) {
                            description = new StringBuilder(512)
                                .append("Cannot cancel metrics deletion, because it may lead to undesired results. ")
                                .append("Restored metrics may be accidentally deleted by near TTL cycles. ")
                                .append("Increase TTL in configuration for following shards and try again.");
                            processedNumIds = new IntOpenHashSet(4);
                        }

                        if (processedNumIds.add(numId)) {
                            description
                                .append(" Shard ").append(shardConf.getId())
                                .append(" has TTL equal to ").append(metricsTtlDays).append(" days, ")
                                .append("but min last point in deleted metrics is ").append(minLastPoint).append(".");
                        }
                    }
                }
            }
        }

        return description == null
            ? null
            : description.toString();
    }
}
