package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.gateway.api.task.DeleteMetricsProgress.PhaseOnReplicaProgress;
import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest.Phase;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.gateway.tasks.ScatterGather;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.util.future.RetryConfig;

import static ru.yandex.solomon.gateway.tasks.deleteMetrics.DeleteMetricsTaskProto.remoteParams;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class PhaseOnReplica implements AutoCloseable {

    private final String clusterId;
    private final ScatterGather<RemoteTaskProgress> sg;

    PhaseOnReplica(
        Phase phase,
        RetryConfig retry,
        CoremonClient coremonClient,
        Executor executor,
        ScheduledExecutorService timer,
        String operationId,
        DeleteMetricsParams params,
        PhaseOnReplicaProgress progressOnReplica)
    {
        this.clusterId = progressOnReplica.getClusterId();

        var request = DeleteMetricsRequest.newBuilder()
            .setOperationId(operationId)
            .setSelectors(params.getSelectors())
            .setPhase(phase)
            .setCreatedAt(params.getCreatedAt());

        var initialProgressOnShard = RemoteTaskProgress.newBuilder()
            .setClusterId(progressOnReplica.getClusterId())
            .build();

        var progressOnShardByNumId = progressOnShardByNumId(progressOnReplica);

        var shards = new ArrayList<PhaseOnShard>(params.getNumIdsCount());
        for (int i = 0; i < params.getNumIdsCount(); i++) {
            var numId = params.getNumIds(i);

            var progressOnShard = progressOnShardByNumId.getOrDefault(numId, initialProgressOnShard);

            var shard = new PhaseOnShard(
                DeleteMetricsTaskHandler.TYPE + "_" + phase,
                retry,
                coremonClient,
                executor,
                timer,
                request.setNumId(numId).build(),
                progressOnShard);
            shards.add(shard);
        }

        this.sg = new ScatterGather<>(shards);
    }

    public CompletableFuture<Void> start(boolean interrupted) {
        return sg.start(interrupted);
    }

    public PhaseOnReplicaProgress progress() {
        return PhaseOnReplicaProgress.newBuilder()
            .setClusterId(clusterId)
            .addAllOnShards(sg.progress())
            .build();
    }

    @Override
    public void close() {
        sg.close();
    }

    private static Int2ObjectMap<RemoteTaskProgress> progressOnShardByNumId(PhaseOnReplicaProgress progress) {
        if (progress.getOnShardsList().isEmpty()) {
            return Int2ObjectMaps.emptyMap();
        }

        var progressByNumId = new Int2ObjectOpenHashMap<RemoteTaskProgress>();
        for (var progressOnShard : progress.getOnShardsList()) {
            if (progressOnShard.hasRemoteTask()) {
                var numId = extractNumId(progressOnShard.getRemoteTask());
                progressByNumId.put(numId, progressOnShard);
            }
        }
        return progressByNumId;
    }

    private static int extractNumId(Task remoteTask) {
        var remoteParams = remoteParams(remoteTask.getParams());
        return remoteParams.getNumId();
    }
}
