package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.gateway.api.task.RemoteTaskProgress;
import ru.yandex.gateway.api.task.RemoteTaskProgressOrBuilder;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.DeleteMetricsResponse;
import ru.yandex.solomon.coremon.client.CoremonClient;
import ru.yandex.solomon.gateway.tasks.RemoteTask;
import ru.yandex.solomon.gateway.tasks.SubTask;
import ru.yandex.solomon.scheduler.proto.GetTaskRequest;
import ru.yandex.solomon.scheduler.proto.Task;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class PhaseOnShard implements SubTask<RemoteTaskProgress> {

    private final RemoteTask remoteTask;

    PhaseOnShard(
        String type,
        RetryConfig retry,
        CoremonClient coremonClient,
        Executor executor,
        ScheduledExecutorService timer,
        DeleteMetricsRequest request,
        RemoteTaskProgress progress)
    {
        this.remoteTask = new RemoteTask(
            type,
            executor,
            timer,
            progress,
            new DeleteMetricsTaskClient(retry, coremonClient, progress.getClusterId(), request),
            PhaseOnShard::isComplete);
    }

    private static boolean isComplete(RemoteTaskProgressOrBuilder progress) {
        return progress.getComplete() || progress.getRemoteTaskCompletedAt() > 0;
    }

    @Override
    public CompletableFuture<Void> start(boolean interrupt, Runnable onProgressChange) {
        return remoteTask.start(interrupt, onProgressChange);
    }

    @Override
    public boolean isIdle() {
        return remoteTask.isIdle();
    }

    @Override
    public RemoteTaskProgress progress() {
        return remoteTask.progress();
    }

    @Override
    public void close() {
        remoteTask.close();
    }

    private static class DeleteMetricsTaskClient extends RemoteTask.RemoteTaskClient {
        private final CoremonClient coremonClient;
        private final String clusterId;
        private final DeleteMetricsRequest request;

        DeleteMetricsTaskClient(
            RetryConfig retry,
            CoremonClient coremonClient,
            String clusterId,
            DeleteMetricsRequest request)
        {
            super(retry);
            this.coremonClient = coremonClient;
            this.clusterId = clusterId;
            this.request = request;
        }

        @Override
        protected CompletableFuture<String> interrupt() {
            return deleteMetrics(request.toBuilder().setInterrupt(true).build());
        }

        @Override
        protected CompletableFuture<String> schedule() {
            return deleteMetrics(request);
        }

        @Override
        protected CompletableFuture<Task> getTask(GetTaskRequest request) {
            return retry(() -> coremonClient.getTask(clusterId, request));
        }

        private CompletableFuture<String> deleteMetrics(DeleteMetricsRequest request) {
            return retry(() -> coremonClient.deleteMetrics(clusterId, request))
                .thenApply(DeleteMetricsResponse::getTaskId);
        }
    }
}
