package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.solomon.gateway.operations.LongRunningOperation;
import ru.yandex.solomon.gateway.operations.deleteMetrics.DeleteMetricsOperationManager;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class PrepareOperation implements AutoCloseable {

    private static final Logger logger = LoggerFactory.getLogger(PrepareOperation.class);

    private final RetryContext retryCtx;

    private final String operationId;
    private final DeleteMetricsParams params;

    private final DeleteMetricsOperationManager manager;

    PrepareOperation(
        RetryConfig retry,
        DeleteMetricsOperationManager manager,
        String operationId,
        DeleteMetricsParams params)
    {
        this.retryCtx = new RetryContext(retry);
        this.manager = manager;
        this.operationId = operationId;
        this.params = params;
    }

    public CompletableFuture<LongRunningOperation> prepare() {
        try {
            return tryPrepare();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<LongRunningOperation> tryPrepare() {
        return tryCreateOperation();
    }

    private CompletableFuture<LongRunningOperation> tryCreateOperation() {
        return retry(
            () -> manager.tryCreateOperation(operationId, params)
                .thenCompose(result -> {
                    if (result.isOk()) {
                        return completedFuture(result.operation());
                    }

                    if (result.isAlreadyExists()) {
                        logger.info("delete metrics operation already created: {}", operationId);
                        return completedFuture(result.operation());
                    }

                    return failedFuture(result.status().asRuntimeException());
                }));
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryCtx.retry(supplier);
    }

    @Override
    public void close() {
        retryCtx.close();
    }
}
