package ru.yandex.solomon.gateway.tasks.deleteMetrics;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Status;

import ru.yandex.gateway.api.task.DeleteMetricsParams;
import ru.yandex.gateway.api.task.DeleteMetricsProgress;
import ru.yandex.solomon.scheduler.grpc.Proto;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
record ProgressEstimation(int estimatedMetrics, double progressPercentage, List<ProgressStatus> progressStatuses) {

    ProgressEstimation {
        checkArgument(progressPercentage >= 0 && progressPercentage <= 100);
        checkArgument(estimatedMetrics >= 0);
        checkNotNull(progressStatuses);
    }

    static ProgressEstimation onRollback(DeleteMetricsParams params, DeleteMetricsProgress progress) {
        var estimatedMetrics = 0;
        var replicaProgressSum = 0.0;
        var remoteStatuses = List.<ProgressStatus>of();

        for (var replica : progress.getRollbackOnReplicasList()) {
            var replicaEstimatedMetrics = 0;
            var shardProgressSum = 0.0;

            for (var shard : replica.getOnShardsList()) {
                if (shard.hasRemoteTask()) {
                    var remoteTask = shard.getRemoteTask();

                    var remoteProgress = DeleteMetricsTaskProto.remoteRollbackProgress(remoteTask.getProgress());
                    var rollbackDeletedMetrics = remoteProgress.getRollbackDeletedMetrics();

                    replicaEstimatedMetrics += rollbackDeletedMetrics.getTotalMetrics();
                    shardProgressSum += rollbackDeletedMetrics.getProgress();

                    var remoteProgressStatus = Proto.fromProto(remoteProgress.getStatus());

                    if (!remoteProgressStatus.isOk()) {
                        if (remoteStatuses.isEmpty()) {
                            remoteStatuses = new ArrayList<>(4);
                        }

                        var numId = DeleteMetricsTaskProto.remoteParams(remoteTask.getParams()).getNumId();
                        remoteStatuses.add(new ProgressStatus(replica.getClusterId(), numId, remoteProgressStatus));
                    }
                }
            }

            estimatedMetrics = Math.max(estimatedMetrics, replicaEstimatedMetrics);
            replicaProgressSum += shardProgressSum / params.getNumIdsCount();
        }

        var progressPercentage = Math.min(99, 100 * replicaProgressSum / progress.getRollbackOnReplicasCount());

        return new ProgressEstimation(estimatedMetrics, progressPercentage, remoteStatuses);
    }

    static ProgressEstimation onMove(DeleteMetricsParams params, DeleteMetricsProgress progress) {
        var estimatedMetrics = 0;
        var replicaProgressSum = 0.0;

        for (var replica : progress.getMoveOnReplicasList()) {
            var replicaEstimatedMetrics = 0;
            var shardProgressSum = 0.0;

            for (var shard : replica.getOnShardsList()) {
                if (shard.hasRemoteTask()) {
                    var remoteTask = shard.getRemoteTask();
                    if (shard.getComplete()) {
                        var remoteResult = DeleteMetricsTaskProto.remoteMoveResult(remoteTask.getResult());

                        replicaEstimatedMetrics += remoteResult.getMovedMetrics();
                        shardProgressSum += 1;
                    } else {
                        var remoteProgress = DeleteMetricsTaskProto.remoteMoveProgress(remoteTask.getProgress());
                        var moveToDeleted = remoteProgress.getMoveToDeletedMetrics();

                        replicaEstimatedMetrics += moveToDeleted.getEstimatedTotalMetrics();
                        shardProgressSum += moveToDeleted.getProgress();
                    }
                }
            }

            estimatedMetrics = Math.max(estimatedMetrics, replicaEstimatedMetrics);
            replicaProgressSum += shardProgressSum / params.getNumIdsCount();
        }

        var progressPercentage = Math.min(99, 100 * replicaProgressSum / progress.getMoveOnReplicasCount());

        return new ProgressEstimation(estimatedMetrics, progressPercentage, List.of());
    }

    record ProgressStatus(String clusterId, int numId, Status status) {
        ProgressStatus {
            checkNotNull(clusterId);
            checkArgument(numId != 0);
            checkNotNull(status);
        }
    }
}
